/* Link-cell routines for molecular modeling package. */

#include "build.h"
#include "ff.h"
#include "proto.h"

/* Set up data structures for cell-search algorithm. */
void set_up_cells_phantom(double **h, int *nc, int *nc_p, int *nc_p_total, 
    atom_cell *atom_cells, int kc, int period_switch, double r_nl, 
    double r_off, int neigh_switch, int **first, int **last, int *offsets, 
    int n_atoms, double **scaled_atom_coords, double **atom_coords, 
    phantom_cell **phantoms, int *phantom_skip)
{
   int nc_p_old[NDIM], nc_p_total_old, cell_flag_1, cell_flag_2;
   int k, n;
   int i, ic_old;

   /* Save previous number of cells. */
   for (k = 0; k < NDIM; ++k )
        nc_p_old[k] = nc_p[k];
   nc_p_total_old = *nc_p_total;

   /* Calculate number of "real" cells. */
   if (neigh_switch == 2) {
      for (k = 0; k < NDIM; ++k) 
         nc[k] = (int) (kc * h[k][k] / r_nl);
   }
   else {
      for (k = 0; k < NDIM; ++k) 
         nc[k] = (int) (kc * h[k][k] / r_off);
   }

   /* Don't take into account 2 times the same atoms (i.e real + image): 
      the number of cells must be at least 2 kc + 1. */
   for ( k = 0; k < NDIM; ++k) {
      if (nc[k] < 2 * kc + 1)
          error_exit("nc < 2 * kc + 1 in set_up_cells");
   }

   /* Calculate number of cells (real cells plus "phantom" or
      "padding" cells) in each direction. */
    for (k = 0; k < NDIM; ++k) {
      nc_p[k] = nc[k] + 2 * kc;
    }

   /* Calculate total number of cells (real cells plus "phantom" or
      "padding" cells). */
   *nc_p_total = nc_p[0] * nc_p[1] * nc_p[2];

   /* Set cell data structure update flags. */
   cell_flag_1 = *nc_p_total != nc_p_total_old;
   cell_flag_2 = nc_p[0] != nc_p_old[0] || nc_p[1] != nc_p_old[1]
                 || nc_p[2] != nc_p_old[2];
   
   /* Set up array of cell offsets. */
   if (cell_flag_2)
      set_up_offsets(kc, nc_p, offsets);

   /* If period_switch == 1, set up arrays used in creating phantoms. */
   if (period_switch)
      set_up_phantom_arrays(cell_flag_1, nc_p, *nc_p_total, phantoms, kc, 
         nc, h);

   /* Allocate memory for labels of the first and last atoms in cell. */
   if (cell_flag_1) {
      if ((*first = realloc(*first, *nc_p_total * sizeof(int))) == NULL)
         error_exit("Out of memory in set_up_cells");
      if ((*last = realloc(*last, *nc_p_total * sizeof(int))) == NULL)
         error_exit("Out of memory in set_up_cells");
   }

   /* Initialize cell lists. */
   if (cell_flag_2)
       initialize_cells(n_atoms, period_switch, h, atom_coords, 
          scaled_atom_coords, *first, *last, nc, nc_p, *nc_p_total, kc,
          atom_cells, *phantoms, phantom_skip); 

}
/*****************************************************************************/ 
/* Set up array of cell offsets. */
void set_up_offsets(int kc, int *nc_p, int *offsets)
{
   int icnt, ix, iy, iz, ic;

   /* Set up array of cell offsets used in computing forces. */
   icnt = 0;
   for (iz = -kc; iz <= kc; ++iz)
      for (iy = -kc; iy <= kc; ++iy)
         for (ix = -kc; ix <= kc; ++ix) {
            ic = ix + nc_p[0] * (iy + nc_p[1] * iz);
            offsets[icnt++] = ic;
         }
}
/*****************************************************************************/
/* Set up arrays used in creating phantoms. */
void set_up_phantom_arrays(int cell_flag_1, int *nc_p, int nc_p_total,
    phantom_cell **phantoms, int kc, int *nc, double **h)
{
   int ic, rx, ry, rz;

   /* Allocate memory for phantom arrays. */
   if (cell_flag_1)
      if ((*phantoms = 
                realloc(*phantoms, nc_p_total * sizeof(phantom_cell))) == NULL)
         error_exit("Out of memory in set_up_phantom_arrays");

   /* Zero phantom counters. */
   for (ic = 0; ic < nc_p_total; ++ic)
      (*phantoms)[ic].n = 0;

   /* Loop over blocks of phantom cells. */
   for (rx = -1; rx <= 1; ++rx)
      for (ry = -1; ry <= 1; ++ry)
         for (rz = -1; rz <= 1; ++rz)
            if (!(rx == 0 && ry == 0 && rz == 0))
               phantom_block(rx, ry, rz, nc, nc_p, kc, h, *phantoms);
}

/*****************************************************************************/
/* Set up phantom arrays for a block of cells. */
void phantom_block(int rx, int ry, int rz, int *nc, int *nc_p, int kc,
   double **h, phantom_cell *phantoms)
{
   int ix, ix1, ix2, iy, iy1, iy2, iz, iz1, iz2, d_ix, d_iy, d_iz,
      ic, ic_p;
   double d_x, d_y, d_z;

   if (rx == -1)
      ix1 = nc[0], ix2 = kc + nc[0] - 1;
   else if (rx == 0)
      ix1 = kc, ix2 = kc + nc[0] - 1;
   else
      ix1 = kc, ix2 = 2 * kc - 1;
   if (ry == -1)
      iy1 = nc[1], iy2 = kc + nc[1] - 1;
   else if (ry == 0)
      iy1 = kc, iy2 = kc + nc[1] - 1;
   else
      iy1 = kc, iy2 = 2 * kc - 1;
   if (rz == -1)
      iz1 = nc[2], iz2 = kc + nc[2] - 1;
   else if (rz == 0)
      iz1 = kc, iz2 = kc + nc[2] - 1;
   else
      iz1 = kc, iz2 = 2 * kc - 1;
   d_ix = rx * nc[0];
   d_iy = ry * nc[1];
   d_iz = rz * nc[2];
   d_x = rx * h[0][0];
   d_y = ry * h[1][1];
   d_z = rz * h[2][2];
   for (iz = iz1; iz <= iz2; ++iz)
      for (iy = iy1; iy <= iy2; ++iy)
         for (ix = ix1; ix <= ix2; ++ix) {
            ic = ix + nc_p[0] * (iy + nc_p[1] * iz);
            ic_p = ix + d_ix + nc_p[0] * (iy + d_iy + nc_p[1] * (iz + d_iz));
            phantoms[ic_p].d_x = d_x;
            phantoms[ic_p].d_y = d_y;
            phantoms[ic_p].d_z = d_z;
            phantoms[ic].ic[phantoms[ic].n++] = ic_p;
         }
}

/*****************************************************************************/
/* Initialize cell lists. */

void initialize_cells(int n_atoms, int period_switch, double **h, 
    double **atom_coords, double **scaled_atom_coords, int *first, int *last,
    int *nc, int *nc_p, int nc_p_total, int kc, atom_cell *atom_cells,
    phantom_cell *phantoms, int *phantom_skip)

{
   int i, ic, i_phant, i_p, ic_p;

   /* If period_switch == 1, apply periodic boundary conditions. */
   if (period_switch)
     periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);

   /* Purge cell lists. */
   for (ic = 0; ic < nc_p_total; ++ic) {
      first[ic] = -1;
      last[ic] = -1;
   }
   for (i = 0; i < n_atoms; ++i) {
      atom_cells[i].prev = -1;
      atom_cells[i].next = -1;
      atom_cells[i].cell = -1;
      atom_cells[i].real = i;
   }
   if (period_switch) {
      for (i = n_atoms; i < 8 * n_atoms; ++i) {
         atom_cells[i].prev = -1;
         atom_cells[i].next = -1;
         atom_cells[i].cell = -1;
         atom_cells[i].real = -1;
      }
   }

   /* Assign atoms to cells. */
   for (i = 0; i < n_atoms; ++i) {

      /* Calculate cell index for atom i. */
      if (period_switch)
         ic = cell_index_period(scaled_atom_coords[i], nc, nc_p, kc);

      /* Add entry to real cell list. */
      add_entry(i, ic, atom_cells, first, last);

      /* If period_switch == 1, add entries to phantom cell lists. */
      if (period_switch)
         for (i_phant = 0; i_phant < phantoms[ic].n; ++i_phant) {
            i_p = i + phantom_skip[i_phant];
            ic_p = phantoms[ic].ic[i_phant];
            add_entry(i_p, ic_p, atom_cells, first, last);
            atom_cells[i_p].real = i;
         }

   }
}

/*****************************************************************************/
/* Calculate cell index for atom i, with periodic boundary conditions. */
int cell_index_period(double *scaled_coords, int *nc, int *nc_p, int kc)
{
   int ix, iy, iz, ic;

   ix = (int) ((scaled_coords[0] + 0.5) * nc[0]) + kc;
   iy = (int) ((scaled_coords[1] + 0.5) * nc[1]) + kc;
   iz = (int) ((scaled_coords[2] + 0.5) * nc[2]) + kc;
   /* Avoid to go out of range when x=0.5 (and/or y=0.5, and/or z=0.5).
      Here we chose to fold it back to the previous cell index in the row. */
   if (scaled_coords[0] == 0.5) ix -=1;
   if (scaled_coords[1] == 0.5) iy -=1;
   if (scaled_coords[2] == 0.5) iz -=1;

   /* Avoid to go out of range when x=0.5 (and/or y=0.5, and/or z=0.5).
      Here we chose to fold it back to the first REAL cell index in the row. */
/* if (ix > nc[0] - 1) ix = kc;
   if (iy > nc[1] - 1) iy = kc;
   if (iz > nc[2] - 1) iz = kc;
*/
   ic = ix + nc_p[0] * (iy + nc_p[1] * iz);

   return ic;
}

/*****************************************************************************/
/* Add entry to cell lists. */
void add_entry(int i, int ic, atom_cell *atom_cells, int *first, int *last)
{
   int i_first, i_last;

   /* Assign new cell index to atom i. */
   atom_cells[i].cell = ic;

   /* If the cell list is empty, put atom i at the head of the list. 
      Otherwise, append atom i to the end of the list. */
   i_first = first[ic];
   if (i_first == -1) {
      first[ic] = i;
      last[ic] = i;
   } 
   else {
      i_last = last[ic];
      last[ic] = i;
      atom_cells[i_last].next = i;
      atom_cells[i].prev = i_last;
   }
}

/*****************************************************************************/
/* Remove entry from cell lists. */
void remove_entry(int i, int ic, atom_cell *atom_cells, int *first, int *last)
{
   int i_prev, i_next;

   /* Get labels of previous and next atoms in cell list. */
   i_prev = atom_cells[i].prev;
   i_next = atom_cells[i].next;

   /* Reset next pointer for previous atom and prev pointer for next atom
      and/or reset pointers to head and tail of list if appropriate. */
   if (i_prev == -1)
      first[ic] = i_next;
   else
      atom_cells[i_prev].next = i_next;
   if (i_next == -1)
      last[ic] = i_prev;
   else
      atom_cells[i_next].prev = i_prev;

   /* Disconnect atom i from list. */
   atom_cells[i].prev = -1;
   atom_cells[i].next = -1;
   atom_cells[i].cell = -1;
}

/*****************************************************************************/
/* Update positions of all phantom atoms. */
void update_phantoms(int n_atoms, atom_cell *atom_cells, int *phantom_skip,
                     phantom_cell *phantoms, double **atom_coords)
{
   int i, ic, i_phant, i_p, ic_p;

   /* Loop over phantom atoms and update their positions. */
   for (i = 0; i < n_atoms; ++i) {
      ic = atom_cells[i].cell;
      for (i_phant = 0; i_phant < phantoms[ic].n; ++i_phant) {
         i_p = i + phantom_skip[i_phant];
         ic_p = phantoms[ic].ic[i_phant];
         atom_coords[i_p][0] = atom_coords[i][0] + phantoms[ic_p].d_x;
         atom_coords[i_p][1] = atom_coords[i][1] + phantoms[ic_p].d_y;
         atom_coords[i_p][2] = atom_coords[i][2] + phantoms[ic_p].d_z;
      }
   }
}

/*****************************************************************************/
/* Update positions of phantom atoms associated with a single molecule. */
void update_phantoms_single(int i_mol, int n_atoms, atom_cell *atom_cells,
    int *phantom_skip, phantom_cell *phantoms, double **atom_coords, 
    int *mol_first_atm, int *mol_species, int *n_atoms_per_mol)
{
   int i_species, i, ic, i_phant, i_p, ic_p, skip;

   /* Get label of molecular species and label of first atom in molecule. */
   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];

   /* Loop over phantom atoms and update their positions. */
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {
      ic = atom_cells[i].cell;
      for (i_phant = 0; i_phant < phantoms[ic].n; ++i_phant) {
         i_p = i + phantom_skip[i_phant];
         ic_p = phantoms[ic].ic[i_phant];
         atom_coords[i_p][0] = atom_coords[i][0] + phantoms[ic_p].d_x;
         atom_coords[i_p][1] = atom_coords[i][1] + phantoms[ic_p].d_y;
         atom_coords[i_p][2] = atom_coords[i][2] + phantoms[ic_p].d_z;
      }
   }
}

/*****************************************************************************/
/* Update cell lists for all atoms. */
void update_cells(int period_switch, int n_atoms, double **h,
    double **scaled_atom_coords, double **atom_coords, atom_cell *atom_cells, 
    int *first, int *last, phantom_cell *phantoms, int kc, int *phantom_skip, 
    int *nc, int *nc_p)
{
   int i, ic_old, ic, i_phant, i_p, ic_p_old, ic_p;

   /* If params.period_switch == 1, apply periodic boundary conditions. */
/*   if (period_switch)
     periodic_boundary_conditions(n_atoms, h, scaled_atom_coords, atom_coords);
*/

   /* Update cell lists as necessary. */
   for (i = 0; i < n_atoms; ++i) {

      /* Save previous cell index. */
      ic_old = atom_cells[i].cell;

      /* Calculate cell index for atom i. */
      if (period_switch)
        ic = cell_index_period(scaled_atom_coords[i], nc, nc_p, kc);

      /* if ic != ic_old, update cell lists. */
      if (ic != ic_old) {

         /* Update real cells. */
         remove_entry(i, ic_old, atom_cells, first, last);
         add_entry(i, ic, atom_cells, first, last);

         /* If period_switch == 1, update phantom cells. */
         if (period_switch) {
            for (i_phant = 0; i_phant < phantoms[ic_old].n; ++i_phant) {
               i_p = i + phantom_skip[i_phant];
               ic_p_old = phantoms[ic_old].ic[i_phant];
               remove_entry(i_p, ic_p_old, atom_cells, first, last);
               atom_cells[i_p].real = -1;
            }
            for (i_phant = 0; i_phant < phantoms[ic].n; ++i_phant) {
               i_p = i + phantom_skip[i_phant];
               ic_p = phantoms[ic].ic[i_phant];
               add_entry(i_p, ic_p, atom_cells, first, last);
               atom_cells[i_p].real = i;
            }
         }
      }
   }
}
/*****************************************************************************/
/* Update cell lists for a single molecule. */
void update_cells_single(int i_mol, int period_switch, int n_atoms, double **h,
    double **scaled_atom_coords, double **atom_coords, atom_cell *atom_cells, 
    int *first, int *last, phantom_cell *phantoms, int kc, int *phantom_skip, 
    int *nc, int *nc_p, int *n_atoms_per_mol, int *mol_species, 
    int *mol_first_atm)
{
   int i_species, i, ic_old, ic, i_phant, skip, i_p, ic_p_old, ic_p;

   /* If period_switch == 1, apply periodic boundary conditions. */
 /*  if (period_switch)
      periodic_boundary_conditions_single(i_mol, n_atoms_per_mol,
            mol_first_atm, mol_species, h, atom_coords, scaled_atom_coords);
*/
   /* Get label of molecular species and label of first atom in molecule. */
   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];

   /* Update cell lists as necessary. */
   for (i = skip; i < skip + n_atoms_per_mol[i_species]; ++i) {

      /* Save previous cell index. */
      ic_old = atom_cells[i].cell;

      /* Calculate cell index for atom i. */
      if (period_switch)
         ic = cell_index_period(scaled_atom_coords[i], nc, nc_p, kc);

      /* if ic != ic_old, update cell lists. */
      if (ic != ic_old) {

         /* Update real cells. */
         remove_entry(i, ic_old, atom_cells, first, last);
         add_entry(i, ic, atom_cells, first, last);

      /* If params.period_switch == 1, update phantom cells. */
      if (period_switch) {
        for (i_phant = 0; i_phant < phantoms[ic_old].n; ++i_phant) {
           i_p = i + phantom_skip[i_phant];
           ic_p_old = phantoms[ic_old].ic[i_phant];
           remove_entry(i_p, ic_p_old, atom_cells, first, last);
           atom_cells[i_p].real = -1;
        }
        for (i_phant = 0; i_phant < phantoms[ic].n; ++i_phant) {
           i_p = i + phantom_skip[i_phant];
           ic_p = phantoms[ic].ic[i_phant];
           add_entry(i_p, ic_p, atom_cells, first, last);
           atom_cells[i_p].real = i;
        }
      }
      }
   }
}
/*****************************************************************************/
