/* Spatial correlation function routines. */

#include "build.h"

/* Calculate center of mass pair correlation functions. */
void spatial_correlation(int n_r, int n_r_par, int n_r_perp, 
   double one_over_r_incr, double one_over_r_par_incr,
   double one_over_r_perp_incr, double r2_max, double r_par_max,
   int n_mol,
   double **scaled_mol_coords, double ***mol_inert_axis, double volume, 
   double **h, double *g_r, double *g_1, double *g_2, double **g_r_par_perp, 
   double **g_1_par_perp, double **g_2_par_perp, int *mol_first_atm,
   double **scaled_atom_coords)

{
   int i_mol, j_mol, i_r, i_r_par, i_r_perp, k, count, skip_i, skip_j;
   double r2, r, r_par, x_perp, y_perp, z_perp, r_perp,
      u_i_dot_u_j, pu_i_dot_pu_j, p_1, p_2, s_sep[3], sep[3],
      pu_i[3], pu_j[3], u1_i[3], u2_i[3], pu[3], s1[3], s2[3], s1_i[3], s2_i[3],
      s1_j[3], s2_j[3], norm1, norm2, u1_j[3], u2_j[3];
    count = 0;

   /* Loop over all pairs of molecules. */
   for (i_mol = 0; i_mol < n_mol - 1; ++i_mol)
      for (j_mol = i_mol + 1; j_mol < n_mol; ++j_mol) {
       

     /* Get the label of the central atom for i_mol and j_mol */
       skip_i = mol_first_atm[i_mol];
       skip_j = mol_first_atm[j_mol];
       

       /* Calculate separation of molecules i_mol and j_mol. */
       for (k = 0; k < 3; ++k){
          s_sep[k] = scaled_mol_coords[i_mol][k] - scaled_mol_coords[j_mol][k]; 
          s_sep[k] -= NINT(s_sep[k]);
          sep[k] = s_sep[k] * h[k][k];
       }

       r2 = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);


       /* Test whether squared pair separation is within cutoff. */
       if (r2 < r2_max) {

          /* Calculate separation of molecules i_mol and j_mol. */
          r = sqrt(r2);

          /* Calculate molecular directors for molecules i_mol and j_mol,
          as well as Legendre polynomials of their dot product. */
    /* calculate the two unit vector in the plane of the i-th molecule */
      for (k = 0; k < NDIM; ++k){
         s1_i[k] = scaled_atom_coords[skip_i][k] - scaled_atom_coords[skip_i+7][k];
         s2_i[k] = scaled_atom_coords[skip_i][k] - scaled_atom_coords[skip_i+1][k];
         s1_i[k] -= NINT(s1_i[k]);
         s2_i[k] -= NINT(s2_i[k]);
         s1[k] = s1_i[k] * h[k][k];
         s2[k] = s2_i[k] * h[k][k];
      }
         norm1 =  SQR(s1[0]) + SQR(s1[1]) + SQR(s1[2]);
         norm2 =  SQR(s2[0]) + SQR(s2[1]) + SQR(s2[2]);

      /* Compute two unit vecore fixed on molecule i_mol */
       for ( k = 0; k < NDIM; ++k) {
            u1_i[k] = s1[k]/sqrt(norm1);
            u2_i[k] = s2[k]/sqrt(norm2);
       }

    /* calculate the two unit vector in the plane of the j-th molecule */
      for (k = 0; k < NDIM; ++k){
         s1_j[k] = scaled_atom_coords[skip_j][k] - scaled_atom_coords[skip_j+7][k];
         s2_j[k] = scaled_atom_coords[skip_j][k] - scaled_atom_coords[skip_j+1][k];
         s1_j[k] -= NINT(s1_j[k]);
         s2_j[k] -= NINT(s2_j[k]);
         s1[k] = s1_j[k] * h[k][k];
         s2[k] = s2_j[k] * h[k][k];
      }
         norm1 =  SQR(s1[0]) + SQR(s1[1]) + SQR(s1[2]);
         norm2 =  SQR(s2[0]) + SQR(s2[1]) + SQR(s2[2]);

      /* Compute two unit vecore fixed on molecule i_mol */
       for ( k = 0; k < NDIM; ++k) {
            u1_j[k] = s1[k]/sqrt(norm1);
            u2_j[k] = s2[k]/sqrt(norm2);
       }

      /* Calculate the cross product of u1_i and u2_i to get the unit vector 
         perpendicular to them */
       pu_i[0] = u1_i[0] * u2_i[2] - u1_i[2] * u2_i[0];
       pu_i[1] = u1_i[2] * u2_i[1] - u1_i[1] * u2_i[2];
       pu_i[2] = u1_i[0] * u2_i[1] - u1_i[1] * u2_i[0];
        
       pu_j[0] = u1_j[0] * u2_j[2] - u1_j[2] * u2_j[0];
       pu_j[1] = u1_j[2] * u2_j[1] - u1_j[1] * u2_j[2];
       pu_j[2] = u1_j[0] * u2_j[1] - u1_j[1] * u2_j[0];

          pu_i_dot_pu_j = pu_i[0] * pu_j[0] + pu_i[1] * pu_j[1] + pu_i[2] * pu_j[2];
          p_1 = 1.5 * SQR(pu_i_dot_pu_j) - 0.5; 
          u_i_dot_u_j = u1_i[0] * u1_j[0] + u1_i[1] * u1_j[1] + u1_i[2] * u1_j[2];
          p_2 = 1.5 * SQR(u_i_dot_u_j) - 0.5;


          /* Add contributions to molecular translational and orientational
             correlation functions. */
          i_r = (int) (r * one_over_r_incr);
          if (i_r < n_r) {
          /*     printf(" i_r %d g_r %g\n", i_r, g_r[i_r]);  */
               g_r[i_r] += volume;
               g_1[i_r] += volume * p_1;
               g_2[i_r] += volume * p_2;
          }
          /* Calculate components of displacement of molecule j_mol relative to
          molecule i_mol parallel and perpendicular to director of molecule i_mol. */

          r_par = sep[0] * u1_i[0] + sep[1] * u1_i[1] + sep[2] * u1_i[2];
          r_perp = sqrt(r2 - SQR(r_par));

          /* Add contributions to two-dimensional correlation functions. */

          i_r_par = FLOOR((r_par + r_par_max) * one_over_r_par_incr);
          i_r_perp = (int) (r_perp * one_over_r_perp_incr);
          if (i_r_par >= 0 && i_r_par < n_r_par && i_r_perp < n_r_perp) {
             g_r_par_perp[i_r_par][i_r_perp] += volume;
             g_1_par_perp[i_r_par][i_r_perp] += volume * p_1;
             g_2_par_perp[i_r_par][i_r_perp] += volume * p_2;
          }

          /* Calculate components of displacement of molecule i_mol relative to
          molecule j_mol parallel and perpendicular to director of molecule j_mol. */

          r_par = - sep[0] * u1_j[0] - sep[1] * u1_j[1] - sep[2] * u1_j[2];
          r_perp = sqrt(r2 - SQR(r_par));

          /* Add contributions to two-dimensional correlation functions. */

          i_r_par = FLOOR((r_par + r_par_max) * one_over_r_par_incr);
          i_r_perp = (int) (r_perp * one_over_r_perp_incr);
          if (i_r_par >= 0 && i_r_par < n_r_par && i_r_perp < n_r_perp) {
             g_r_par_perp[i_r_par][i_r_perp] += volume;
             g_1_par_perp[i_r_par][i_r_perp] += volume * p_1;
             g_2_par_perp[i_r_par][i_r_perp] += volume * p_2;
          }
       }
     }

}
