/* Spatial correlation function routines. */

#include "shared.h"

/* Calculate center of mass pair correlation functions. */
void spatial_correlation(int n_r, int n_r_par, int n_r_perp, 
   double one_over_r_incr, double one_over_r_par_incr,
   double one_over_r_perp_incr, double r2_max, double r_par_max,
   int n_mol,
   double **scaled_mol_coords, double ***mol_inert_axis, double volume, 
   double *side, double *g_r, double *g_1, double *g_2, double **g_r_par_perp, 
   double **g_1_par_perp, double **g_2_par_perp)

{
   int i_mol, j_mol, i_r, i_r_par, i_r_perp, k;
   double r2, r, r_par, x_perp, y_perp, z_perp, r_perp,
      u_i_dot_u_j, pu_i_dot_pu_j, p_1, p_2, s_sep[3], sep[3],
      pu_i[3], pu_j[3], u_i[3], u_j[3];


   /* Loop over all pairs of molecules. */
   for (i_mol = 0; i_mol < n_mol - 1; ++i_mol)
      for (j_mol = i_mol + 1; j_mol < n_mol; ++j_mol) {

       /* Calculate separation of molecules i_mol and j_mol. */
       for (k = 0; k < 3; ++k){
          s_sep[k] = scaled_mol_coords[i_mol][k] - scaled_mol_coords[j_mol][k]; 
          s_sep[k] -= NINT(s_sep[k]);
          sep[k] = s_sep[k] * side[k];
       }

       r2 = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

       /* Test whether squared pair separation is within cutoff. */
       if (r2 < r2_max) {

          /* Calculate separation of molecules i_mol and j_mol. */
          r = sqrt(r2);

          /* Calculate molecular directors for molecules i_mol and j_mol,
          as well as Legendre polynomials of their dot product. */

          pu_i[0] = mol_inert_axis[i_mol][1][0];
          pu_i[1] = mol_inert_axis[i_mol][1][1];
          pu_i[2] = mol_inert_axis[i_mol][1][2];

          pu_j[0] = mol_inert_axis[j_mol][1][0];
          pu_j[1] = mol_inert_axis[j_mol][1][1];
          pu_j[2] = mol_inert_axis[j_mol][1][2];

          u_i[0] = mol_inert_axis[i_mol][2][0];
          u_i[1] = mol_inert_axis[i_mol][2][1];
          u_i[2] = mol_inert_axis[i_mol][2][2];

          u_j[0] = mol_inert_axis[j_mol][2][0];
          u_j[1] = mol_inert_axis[j_mol][2][1];
          u_j[2] = mol_inert_axis[j_mol][2][2];

          p_1 = pu_i[0] * pu_j[0] + pu_i[1] * pu_j[1] + pu_i[2] * pu_j[2];
          u_i_dot_u_j = u_i[0] * u_j[0] + u_i[1] * u_j[1] + u_i[2] * u_j[2];
          p_2 = 1.5 * SQR(u_i_dot_u_j) - 0.5;

          /* Add contributions to molecular translational and orientational
             correlation functions. */
          i_r = (int) (r * one_over_r_incr);
          if (i_r < n_r) {
               g_r[i_r] += volume;
               g_1[i_r] += volume * p_1;
               g_2[i_r] += volume * p_2;
          }

          /* Calculate components of displacement of molecule j_mol relative to
          molecule i_mol parallel and perpendicular to director of molecule i_mol. */
          r_par = sep[0] * u_i[0] + sep[1] * u_i[1] + sep[2] * u_i[2];
          /*x_perp = sep.x - r_par * u_i.x;
           y_perp = sep.y - r_par * u_i.y;
           z_perp = sep.z - r_par * u_i.z;
           r_perp = sqrt(SQR(x_perp) + SQR(y_perp) + SQR(z_perp)); */
          r_perp = sqrt(r2 - SQR(r_par));

          /* Add contributions to two-dimensional correlation functions. */
          i_r_par = FLOOR((r_par + r_par_max) * one_over_r_par_incr);
          i_r_perp = (int) (r_perp * one_over_r_perp_incr);
          if (i_r_par >= 0 && i_r_par < n_r_par && i_r_perp < n_r_perp) {
             g_r_par_perp[i_r_par][i_r_perp] += volume;
             g_1_par_perp[i_r_par][i_r_perp] += volume * p_1;
             g_2_par_perp[i_r_par][i_r_perp] += volume * p_2;
          }

          /* Calculate components of displacement of molecule i_mol relative to
          molecule j_mol parallel and perpendicular to director of molecule j_mol. */
          r_par = - sep[0] * u_j[0] - sep[1] * u_j[1] - sep[2] * u_j[2];
          /* x_perp = - sep.x - r_par * u_j.x;
           y_perp = - sep.y - r_par * u_j.y;
           z_perp = - sep.z - r_par * u_j.z;
           r_perp = sqrt(SQR(x_perp) + SQR(y_perp) + SQR(z_perp)); */
          r_perp = sqrt(r2 - SQR(r_par));

          /* Add contributions to two-dimensional correlation functions. */
          i_r_par = FLOOR((r_par + r_par_max) * one_over_r_par_incr);
          i_r_perp = (int) (r_perp * one_over_r_perp_incr);
          if (i_r_par >= 0 && i_r_par < n_r_par && i_r_perp < n_r_perp) {
             g_r_par_perp[i_r_par][i_r_perp] += volume;
             g_1_par_perp[i_r_par][i_r_perp] += volume * p_1;
             g_2_par_perp[i_r_par][i_r_perp] += volume * p_2;
          }
       }
     }

}
