/* Input/output routines for builder module. */

#include "build.h"
#include "ff.h"

/*************************************************************************/
/* Read simulator module input parameter file. */
/*
input :
    name of the file containing the parameters used in the builder (par_file)
output :
    Force field file for masses (mass_file)
    Force field file for potential interactions (pot_file)
    Switch for periodic system (period_switch - 1 means periodic)
    Number of molecular species (n_species)
    Array of number of molecules of each specie in the unit cell 
                                                       (n_mols_per_unit_cell)
    Array of files of the molecular structures (struct_file)
    Switch for the combination rule between atomic site (vdw_rad_switch)
    Overlap criterion between atomi sites (overlap_criterion)
    Switch to chose the initial state being built (init_switch)
    Number of smectic layers present in the initial state (n_layers)
    Layer spacing (layer_spacing)
    Number density for the molecules (rho)
    Number of unit cells (n_unit_cells)
    Aspect ratio for the box (aspect_xy, aspect_xz)
    Array of the molecular principal axe chosen to be along the z cartesian
                                                          axe (principal)
    Seed for the random generator (seed)
    Name of the output header file (header_file)
    Name of the output configuration file (config_file)
*/

void read_energy_params(char *par_file, int *start_switch,
   char **header_file, char **config_file,
   char **mass_file, 
   char **pot_file, int *vdw_rad_switch, int *n_species_chk,
   int **excl_switch, double *r_on, double *r_off, 
   int *neigh_switch, 
   int *phant_switch, int *kc, double *r_nl, int *n_cycles, 
   int *n_block, int *n_trajec, long *seed, double *pe_max, int *n_dist) 
{
   int i_species, i;
   FILE *f_param;
   char line[MAX_LINE];

   /* Open parameter file. */
   if ((f_param = fopen(par_file, "r")) == NULL)
      error_exit("Unable to open parameter file in read_new_build_params");

   /* Read in simu parameters. */
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d \n", start_switch);
   fgets(line, MAX_LINE, f_param);
   *header_file = allocate_1d_array(F_MAX, sizeof(char));
   *config_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s\n", *header_file, *config_file);
   fgets(line, MAX_LINE, f_param);
   *mass_file = allocate_1d_array(F_MAX, sizeof(char));
   *pot_file = allocate_1d_array(F_MAX, sizeof(char));
   fscanf(f_param, "%s %s\n", *mass_file, *pot_file);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", vdw_rad_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d\n", n_species_chk);
   fgets(line, MAX_LINE, f_param);
   *excl_switch = allocate_1d_array(*n_species_chk, sizeof(int));
   for (i = 0; i < *n_species_chk; ++i)
     fscanf(f_param, "%d ", &(*excl_switch)[i]);
   fscanf(f_param, "\n");
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %lf\n", r_on, r_off);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %d\n", neigh_switch, phant_switch);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d %lf\n", kc, r_nl);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%d \n", n_cycles );
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, " %d %d\n", n_block, n_trajec);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%ld\n", seed);
   fgets(line, MAX_LINE, f_param);
   fscanf(f_param, "%lf %d\n", pe_max, n_dist);

   /* Close parameter file. */
   fclose(f_param);
}

/*************************************************************************/

/* Read periodicity switch, number of molecular species, molecular structures,
   numbers of molecules of each species, and list of molecular species to
   header of output file. */
/*
input :
       Name of the input header file
output :
       Switch for periodic system (period_switch)
       Number of species (n_species)
       Array of number of atom in template molecule of each specie 
                                                           (n_atoms_per_mol)
       Array of template atom - atomic labels (temp_atm_lab)
       Array of template atom - atomic types (temp_atm_type)
       Array of template atom - atom branching number (temp_atm_nbr)
       Array of template atom - label of branched atoms (temp_atm_br)
       Array of template atom - order of the bonds branched to an atom
                                                            (temp_atm_ord)
       Array of template atom - realtive position of atoms in the template
                                molecule                    (temp_atm_pos)
       Array of template atom - atomic site charges (temp_atm_chg) 
       Arrays of template atom - LJ atomic site parameters (temp_atm_sigma,
                                                            temp_atm_eps)
       Array of number of bonds in a template molecule (n_bonds_per_mol)
       Arrays of template molecule - label of the 2 atoms sharing a bond
							(temp_bonds_1,
                                                         temp_bonds_2) 
       Array of template molecule - order of a bond (temp_bonds_ord)
       Total number of molecules (n_mols)
       Specie label of the molecules (mol_species)

*/

void read_header_direct(char *header_file, int *period_switch, int *n_species,
  int **n_atoms_per_mol, int ***temp_atm_lab, char ****temp_atm_type, 
  int ***temp_atm_nbr, int ****temp_atm_br, double ****temp_atm_ord, 
  double ****temp_atm_pos, double ***temp_atm_chg, double ***temp_atm_sigma,
  double ***temp_atm_eps, int **n_bonds_per_mol, int ***temp_bonds_1, 
  int ***temp_bonds_2, double ***temp_bonds_ord, int **n_mols_per_species, 
  int *n_mols, int *n_atoms, int **mol_species)
{
   int i, j, i_species;
   FILE *f_input;

   /* Open header file. */
   if ((f_input = fopen(header_file, "r")) == NULL)
      error_exit("Unable to open header file in read_header_direct");

   /* Read in periodicity switch. */
   fscanf(f_input,"%d\n",&(*period_switch));

   /* Read in number of molecular species. */
   fscanf(f_input,"%d\n",&(*n_species));

   /* Allocate memory for arrays containing molecular structures */
   *n_atoms_per_mol = allocate_1d_array(*n_species, sizeof(int));
   *n_mols_per_species = allocate_1d_array(*n_species, sizeof(int));
   *n_bonds_per_mol = allocate_1d_array(*n_species, sizeof(int));
   *temp_atm_lab = allocate_1d_array(*n_species, sizeof(int*));
   *temp_atm_type = allocate_1d_array(*n_species, sizeof(char**));
   *temp_atm_pos = allocate_1d_array(*n_species, sizeof(double**));
   *temp_atm_chg = allocate_1d_array(*n_species, sizeof(double*));
   *temp_atm_nbr = allocate_1d_array(*n_species, sizeof(int*));
   *temp_atm_br = allocate_1d_array(*n_species, sizeof(int**));
   *temp_atm_ord = allocate_1d_array(*n_species, sizeof(double**));
   *temp_bonds_1 = allocate_1d_array(*n_species, sizeof(int*));
   *temp_bonds_2 = allocate_1d_array(*n_species, sizeof(int*));
   *temp_bonds_ord = allocate_1d_array(*n_species, sizeof(double*));
   *temp_atm_sigma = allocate_1d_array(*n_species, sizeof(double*));
   *temp_atm_eps = allocate_1d_array(*n_species, sizeof(double*));

   *n_mols = 0;
   *n_atoms = 0;
   /* Read in molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < *n_species; ++i_species) {

     /* Read in number of atoms per molecule. */
     fscanf(f_input,"%d\n", &(*n_atoms_per_mol)[i_species]);

     /* Allocate memory for template atoms arrays. */
     (*temp_atm_lab)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(int));
     (*temp_atm_type)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], F_MAX, sizeof(char));
     (*temp_atm_chg)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));
     (*temp_atm_nbr)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(int));
     (*temp_atm_pos)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], NDIM, sizeof(double));
     (*temp_atm_br)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], 6, sizeof(int));
     (*temp_atm_ord)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], 6, sizeof(double));
     (*temp_atm_sigma)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));
     (*temp_atm_eps)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));

     /* Read in template atom attributes. */
     for (i = 0; i < (*n_atoms_per_mol)[i_species]; ++i)
      {
      fscanf(f_input,"%d %s %d ", &(*temp_atm_lab)[i_species][i],
         (*temp_atm_type)[i_species][i], &(*temp_atm_nbr)[i_species][i]);
         for (j = 0; j < (*temp_atm_nbr)[i_species][i]; ++j)
          fscanf(f_input,"%d %lf ",&(*temp_atm_br)[i_species][i][j],
            &(*temp_atm_ord)[i_species][i][j]);
         fscanf(f_input,"\n");
         fscanf(f_input,"%lf %lf %lf %lf\n", &(*temp_atm_pos)[i_species][i][0], 
          &(*temp_atm_pos)[i_species][i][1], &(*temp_atm_pos)[i_species][i][2], 
          &(*temp_atm_chg)[i_species][i]);
     /*  fscanf(f_input,"%lf %lf\n", &(*temp_atm_sigma)[i_species][i],
           &(*temp_atm_eps)[i_species][i]); */
      }

     /* Read in number of bonds per molecule. */
     fscanf(f_input,"%d\n", &(*n_bonds_per_mol)[i_species]);

    if ( (*n_bonds_per_mol)[i_species] > 0) {
     (*temp_bonds_1)[i_species] = 
        allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(int));
     (*temp_bonds_2)[i_species] = 
        allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(int));
     (*temp_bonds_ord)[i_species] = 
        allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(double));
   }

     /* Read in template bond attributes. */
     for (i = 0; i < (*n_bonds_per_mol)[i_species]; ++i)
      fscanf(f_input, "%d %d %lf\n", &(*temp_bonds_1)[i_species][i], 
          &(*temp_bonds_2)[i_species][i], &(*temp_bonds_ord)[i_species][i]);

     /* Read in number of molecules. */
     fscanf(f_input, "%d\n", &(*n_mols_per_species)[i_species]);

     /* Increment molecule and atom counters. */
     (*n_mols) += (*n_mols_per_species)[i_species];
     (*n_atoms) += (*n_mols_per_species)[i_species] * 
                               (*n_atoms_per_mol)[i_species]; 
   }

   /* Allocate memory for molecular species array. */
   *mol_species = allocate_1d_array(*n_mols, sizeof(int));

   /* Read in list of molecular species. */
   for (i = 0; i < *n_mols; ++i)
      fscanf(f_input, "%d ",&(*mol_species)[i]);
   fscanf(f_input, "\n");

   /* Close the input file. */
   fclose(f_input);
}

/*************************************************************************/
void write_header_direct(char *header_file, int period_switch, int n_species,
  int *n_atoms_per_mol, int **temp_atm_lab, char ***temp_atm_type,
  int **temp_atm_nbr, int ***temp_atm_br, double ***temp_atm_ord,
  double ***temp_atm_pos, double **temp_atm_chg, double **temp_atm_sigma,
  double **temp_atm_eps, int *n_bonds_per_mol, int **temp_bonds_1,
  int **temp_bonds_2, double **temp_bonds_ord, int *n_mols_per_species,
  int n_mols, int *mol_species)
{
   int i, j, i_species;
   FILE *f_output;

   /* Open header file. */
   if ((f_output = fopen(header_file, "w")) == NULL)
      error_exit("Unable to open header file in write_config_direct");

   /* Write out periodicity switch. */
   fprintf(f_output,"%d\n",period_switch);

   /* Write out number of molecular species. */
   fprintf(f_output,"%d\n",n_species);

   /* Write out molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Write out number of atoms per molecule. */
      fprintf(f_output,"%d\n", n_atoms_per_mol[i_species]);

      /* Write out template atom attributes. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
      {
      fprintf(f_output,"%d %s %d ", temp_atm_lab[i_species][i],
         temp_atm_type[i_species][i], temp_atm_nbr[i_species][i]);
         for (j = 0; j < temp_atm_nbr[i_species][i]; ++j)
          fprintf(f_output,"%d %g ",temp_atm_br[i_species][i][j],
            temp_atm_ord[i_species][i][j]);
         fprintf(f_output,"\n");
         fprintf(f_output,"%g %g %g %g\n", temp_atm_pos[i_species][i][0],
           temp_atm_pos[i_species][i][1], temp_atm_pos[i_species][i][2],
           temp_atm_chg[i_species][i]);
/*      fprintf(f_output,"%g %g\n", temp_atm_sigma[i_species][i],
           temp_atm_eps[i_species][i]); */
      }
      /* Write out number of bonds per molecule. */
      fprintf(f_output,"%d\n", n_bonds_per_mol[i_species]);

      /* Write out template bond attributes. */
      for (i = 0; i < n_bonds_per_mol[i_species]; ++i)
      fprintf(f_output, "%d %d %g\n", temp_bonds_1[i_species][i],
           temp_bonds_2[i_species][i], temp_bonds_ord[i_species][i]);

      /* Write out number of molecules. */
      fprintf(f_output, "%d\n", n_mols_per_species[i_species]);
   }

   /* Write out list of molecular species. */
   for (i = 0; i < n_mols; ++i)
   fprintf(f_output, "%d ", mol_species[i]);
   fprintf(f_output, "\n");

   /* Close header file. */
   fclose(f_output);
}

/*************************************************************************/
/* Read atomic configuration to config file. */
/*
input :
      Name of the configuration input file (config_file)
      Switch about periodicity (period_switch)
      Total number of atoms (n_atoms)
output :
      Array of box matrix (h)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
*/

void read_config_direct(char *config_file, int period_switch, double ***h,
   int n_atoms, double ***atom_coords, double ***atom_vels)
{
   FILE *f_input;
   int i, j;

   /* Open config file. */
   if ((f_input = fopen(config_file, "r")) == NULL)
      error_exit("Unable to open configuration file in read_config_direct");

   /* Write out box matrix if params.period_switch == 1. */
   if (period_switch)
      {
      /* Allocate memory for the box matrix. */
      *h = allocate_2d_array(3, 3, sizeof(double));
      /* Read in box matrix elements. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fscanf(f_input, "%lf ", &(*h)[i][j]);
      fscanf(f_input, "\n");
      }

   /* Allocate memory for arrays of atomic properties. */
   if (period_switch)
      *atom_coords = allocate_2d_array(8 * n_atoms, 3, sizeof(double));
   else
      *atom_coords = allocate_2d_array(n_atoms, 3, sizeof(double));
   *atom_vels = allocate_2d_array(n_atoms, 3, sizeof(double));

   /* Read in atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     fscanf(f_input, "%lf %lf %lf %lf %lf %lf\n",&(*atom_coords)[i][0],
      &(*atom_coords)[i][1], &(*atom_coords)[i][2], &(*atom_vels)[i][0], 
      &(*atom_vels)[i][1], &(*atom_vels)[i][2]);

   /* Close config file. */
   fclose(f_input);
}

/*************************************************************************/
/* Write atomic configuration to config file. */
/*
input :
      Switch about periodicity (period_switch)
      Array of box matrix (h)
      Total number of atoms (n_atoms)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
output :
      Name of the configuration output file (config_file)
*/

void write_config_direct(char *config_file, int period_switch, double **h,
   int n_atoms, double dr_max, double ratio_dang_dr, double dh_max, 
   double skin, double **atom_coords, double **atom_vels, int i_cycle)
{
   FILE *f_output;
   int i, j;
   char par_file[F_MAX];

   /* Open config file. */
/*   sprintf(par_file, "test.config"); */
   if ((f_output = fopen(config_file, "w")) == NULL)
      error_exit("Unable to open configuration file in write_config_direct");

   /* Write out box matrix if params.period_switch == 1. */
   if (period_switch) {
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fprintf(f_output, "%g ", h[i][j]);
      fprintf(f_output, "\n");
    }
   /* Write out MC parameters. */
   fprintf(f_output, "%g %g %g %g %d\n", dr_max, ratio_dang_dr, dh_max, skin, i_cycle);

   /* Write out atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     {
/*     atom_vels[i][0] = atom_vels[i][1] = atom_vels[i][2] = 0.0; */
     fprintf(f_output, "%g %g %g %g %g %g\n", atom_coords[i][0],
      atom_coords[i][1], atom_coords[i][2], atom_vels[i][0], atom_vels[i][1],
      atom_vels[i][2]);

     }
   /* Close config file. */
   fclose(f_output);
}

/****************************************************************************/
/* Read atomic positions from trajec file. */
void read_positions_direct(FILE *f_input, double **h, int n_atoms, 
    double **atom_coords)
{
   int i, j, i_atom;

   /* Read in box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fread(&h[i][j], sizeof(double), 1, f_input);

   /* Read in atomic positions. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
         fread(atom_coords[i_atom], sizeof(double), 3, f_input);
}

/****************************************************************************/
/* Write atomic positions to trajec file. */
void write_positions_direct(FILE *f_output, double **h, int n_atoms, 
    double **atom_coords)
{
   int i, j, i_atom;

   /* Write out box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fwrite(&h[i][j], sizeof(double), 1, f_output);

   /* Write out atomic positions. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
         fwrite(atom_coords[i_atom], sizeof(double), 3, f_output);
}

