/* Mean squared displacement routines. */

#include "build.h"
#include "ff.h"
#include "proto.h"

/* Calculate "unfolded" molecular center of mass positions and write them
   to output file. */
void create_com_pos_file(int period_switch, double **mol_coords_corr, 
                         double **scaled_mol_coords_corr, double **h,
                         double **h_inv, double **atom_coords, 
                         double **scaled_atom_coords, int n_mols, int n_atoms,
                         double **rel_atom_coords, int **scan_atm_1, int **scan_atm_2,
                         double *mass_mass, int *mol_first_atm, int n_species,
                         int *mol_species, int *n_atoms_per_mol, double **mol_coords,
                         double **scaled_mol_coords, char *config_file)
{
   FILE *f_output, *f_trajec;
   int i_data, i_mol, pos_1;
   double s_dx, s_dy, s_dz;
   int k;

   /* Open input and output files. */
   if ((f_trajec = fopen(config_file, "r")) == NULL)
      error_exit("Unable to open positions file in create_com_pos_file");
   if ((f_output = fopen("pro.com_pos_tmp", "w")) == NULL)
      error_exit("Unable to open com_pos file in create_com_pos_file");
   
   /* Position trajectory file pointer at beginning of first data set. */
      fseek(f_trajec, pos_1, 0);

   /* allocate memory for temporary array. */
       mol_coords_corr = allocate_2d_array(n_mols, 3, sizeof(double));
       scaled_mol_coords_corr = allocate_2d_array(n_mols, 3, sizeof(double));

   /* Read in first data set. */
      read_positions_direct(f_trajec, h, n_atoms, atom_coords);

   /* If params.period_switch == 1, calculate quantities that depend
      on box dimensions, calculate scaled atomic coordinates, and apply
      periodic boundary conditions. */
   if (period_switch) {
      box_dimensions(h, h_inv, period_switch, 0.0, 0.0);
      scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
      periodic_boundary_conditions(n_atoms, h, scaled_atom_coords,
                                      atom_coords);
   }

   /* Calculate center of mass positions. */
      center_of_mass_positions(period_switch, n_mols, n_species,
      n_atoms_per_mol,  mol_species, atom_mass, mol_first_atm,
      atom_coords, scaled_atom_coords,  scan_atm_1, scan_atm_2,
      mol_mass, h, h_inv, mol_coords, scaled_mol_coords,  rel_atom_coords);


   /* If period_switch == 1, initialize corrected center of mass
      positions. */
   if (period_switch)
      for (i_mol = 0; i_mol < n_mols; ++i_mol) {
          for ( k = 0; k < NDIM; ++k) {
         scaled_mol_coords_corr[i_mol][k] = scaled_mol_coords[i_mol][k];
         mol_coords_corr[i_mol][k] = mol_coords[i_mol][k];
        }
      }

   /* If period_switch == 1, write out corrected center of mass
      positions. Otherwise, write out "raw" center of mass positions. */
   if (period_switch)
      fwrite(mol_coords_corr, sizeof(vector_3d), consts.n_mols, f_output);
   else
      fwrite(mol_coords, sizeof(vector_3d), consts.n_mols, f_output);

   /* Loop over data sets. */
   for (i_data = 1; i_data < n_snapshots; ++i_data) {

      /* Read in atomic positions. */
      read_positions_direct(f_trajec, h, n_atoms, atom_coords);

      /* If params.period_switch == 1, calculate quantities that depend
         on box dimensions, calculate scaled atomic coordinates, and apply
         periodic boundary conditions. */
      if (period_switch) {
      box_dimensions(h, h_inv, period_switch, 0.0, 0.0);
      scaled_atomic_coords(n_atoms, h_inv, atom_coords, scaled_atom_coords);
      periodic_boundary_conditions(n_atoms, h, scaled_atom_coords,
                                      atom_coords);
      }

      /* Calculate center of mass positions. */
      center_of_mass_positions(period_switch, n_mols, n_species,
      n_atoms_per_mol,  mol_species, atom_mass, mol_first_atm,
      atom_coords, scaled_atom_coords,  scan_atm_1, scan_atm_2,
      mol_mass, h, h_inv, mol_coords, scaled_mol_coords,  rel_atom_coords);


      /* If period_switch == 1, calculate corrected scaled
         center of mass positions. */

      if (period_switch)
         for (i_mol = 0; i_mol < n_mols; ++i_mol) {
             for (k = 0; k <NDIM; ++k){
            sep[k] = scaled_mol_coords[i_mol][k] - scaled_mol_coords_corr[i_mol][k];
            sep[k] -= NINT(sep[k]);
            scaled_mol_coords_corr[i_mol][k] += sep[k];
            mol_coords_corr[i_mol][k] =
               h[k][k] * scaled_mol_coords_corr[i_mol][k];
           }
         }

      /* If period_switch == 1, write out corrected center of mass
         positions. Otherwise, write out "raw" center of mass positions. */
    for ( i_mol = 0; i_mol < n_mols; ++i_mol) {
      if (period_switch)
         fwrite(mol_coords_corr[i_mol], sizeof(double), 3, f_output);
      else
         fwrite(mol_coords[i_mol], sizeof(double), 3, f_output);
    }
   }

   /* Close input and output files. */
   fclose(f_posit);
   fclose(f_output);

   /* If params.period_switch == 1, free up memory. */
   if (period_switch) {
      free(mol_coords_corr);
      free(scaled_mol_coords_corr);
   }
}

/* Calculate mean squared center of mass displacements and write them to
   output file. */
void mean_squared_displacement(double **h, int n_inst, int n_msd, int n_trajec, 
                               int n_mols, double delta, int first_msd)
{
   FILE *f_input, *f_output;
   int pos_1, pos_2, record_size, i0, j0, j0max,
      i_mol, i_msd, offset, *norm, n_fit, i_fit;
   double **mol_coords_corr, **mol_coords_corr_0;
   double dr[NDIM], dr_p[NDIM], dum1, dum2, interval,
      *x, *y, *sig, a, b, chi2, q, siga, sigb, y_fit,
      diffus_x, sig_diffus_x,
      diffus_y, sig_diffus_y,
      diffus_z, sig_diffus_z,
      diffus_xy, sig_diffus_xy,
      diffus_xyz, sig_diffus_xyz,
      **r_displ2, **sig_r_displ2,
      *xy_displ2, *sig_xy_displ2,
      *xyz_displ2, *sig_xyz_displ2;

   /* Allocate memory for and zero center of mass position and mean
      squared center of mass displacement arrays. */

       mol_coords_corr = allocate_2d_array(n_mols, 3, sizeof(double));
       mol_coords_corr_0 = allocate_2d_array(n_mols, 3, sizeof(double));
       r_displ2         = allocate_2d_array(n_msd+1, 3, sizeof(double));
       sig_r_displ2 = allocate_2d_array(n_msd+1, 3, sizeof(double));
       xy_displ2 = allocate_1d_array(n_msd+1, sizeof(double));
       sig_xy_displ2 = allocate_1d_array(n_msd+1, sizeof(double));
       xyz_displ2 = allocate_1d_array(n_msd+1, sizeof(double));
       sig_xyz_displ2 = allocate_1d_array(n_msd+1, sizeof(double));
       norm  = allocate_1d_array(n_msd+1, sizeof(int));

   /* Determine size (in bytes) of records. */
   if ((f_input = fopen("pro.com_pos_tmp", "r")) == NULL)
      error_exit("Unable to open com_pos file in mean_squared_com_displacement");
   pos_1 = ftell(f_input);
   for (i_mol = 0; i_mol < n_mols; ++i_mol)
   fread(mol_coords_corr[i_mol], sizeof(double), 3, f_input);
   pos_2 = ftell(f_input);
   record_size = pos_2 - pos_1;
   fclose(f_input);

   /* Exit if params.n_msd > consts.n_snapshots - 1. */
   if (n_msd > n_inst - 1)
      error_exit("n_msd > n_inst - 1 in mean_squared_com_displacement.");

   /* Reopen input file. */
   if ((f_input = fopen("pro.com_pos_tmp", "r")) == NULL)
      error_exit("Unable to open com_pos file in mean_squared_com_displacement");

   /* Add contributions to mean squared center of mass displacement. */
   for (i0 = 0; i0 < n_inst; ++i0) {

      /* Calculate file position offset for initial data set, and read data
         set starting at that point in the data file. */
      offset = i0 * record_size;
      fseek(f_input, offset, 0);
      for (i_mol = 0; i_mol < n_mols; ++i_mol)
         fread(mol_coords_corr_0[i_mol], sizeof(double), 3, f_input);

      /* Loop over time offsets. */
      j0max = MIN(n_inst - 1, i0 + n_msd);
      for (j0 = i0; j0 <= j0max; ++j0) {

         /* Calculate file position offset for final data set, and read data
            set starting at that point in the data file. */
         offset = j0 * record_size;
         fseek(f_input, offset, 0);
         for (i_mol = 0; i_mol < n_mols; ++i_mol) 
           fread(mol_coords_corr[i_mol], sizeof(double), 3, f_input);

         /* Add contributions to mean squared center of mass displacement. */
         i_msd = j0 - i0;
         for (i_mol = 0; i_mol < n_mols; ++i_mol) {
             for (k = 0; k < NDIM; ++k){
                dr[k] = mol_coords_corr[i_mol][k] - mol_coords_corr_0[i_mol][k];
                dr_p[k] = dr[k] * h[k][k]; 
                r_displ2[i_msd][k] += SQR(dr_p[k]);
             }
         }
         ++norm[i_msd];
      }
   }

   /* Close input file. */
   fclose(f_input);

   /* Normalize mean squared center of mass displacement, calculate
      uncertainties, and write results to output file. Some remarks are
      in order regarding the calculation of uncertainties. The uncertainty
      in <[x(t)-x(0)]^2> is assumed to be equal to the standard deviation
      in [x(t)-x(0)]^2 (which is equal to sqrt(2) * <[x(t)-x(0)]^2> for a
      Gaussian distribution) divided by the square root of the number of
      independent measurements of <[x(t)-x(0)]^2>. Following Allen and
      Tildesley, the number of independent measurements is taken to be
      the total sampling interval divided by twice the correlation
      time for [x(t)-x(0)]^2 , which is estimated to be equal to t (valid
      for large t). We have further assumed that each molecule makes an
      independent contribution to the mean squared displacement. */

   interval = delta * n_trajec;

   f_output = fopen("pro.msd", "w");

   for (i_msd = 0; i_msd <= n_msd; ++i_msd) {
      dum1 = 1.0 / (n_mols * norm[i_msd]);

      for(k = 0; k < NDIM; ++k)
          r_displ2[i_msd][k] *= dum1;

      xy_displ2[i_msd] = r_displ2[i_msd][0] + r_displ2[i_msd][1];
      xyz_displ2[i_msd] = r_displ2[i_msd][0] + r_displ2[i_msd][1] + r_displ2[i_msd][2];
      dum2 = sqrt((4.0 * i_msd) / (n_mols * (n_inst - i_msd)));

      for(k = 0; k < NDIM; ++k)
          sig_r_displ2[i_msd][k] = dum2 * r_displ2[i_msd][k];

      sig_xy_displ2[i_msd] = sqrt(SQR(sig_r_displ2[i_msd][0])
         + SQR(sig_r_displ2[i_msd][1]));

      sig_xyz_displ2[i_msd] = sqrt(SQR(sig_r_displ2[i_msd][0])
         + SQR(sig_r_displ2[i_msd][1]) + SQR(sig_r_displ2[i_msd][2]));

      fprintf(f_output, "%g %g %g %g %g %g %g %g %g %g %g\n",
         i_msd * interval,
         r_displ2[i_msd][0], sig_r_displ2[i_msd][0],
         r_displ2[i_msd][1], sig_r_displ2[i_msd][1],
         r_displ2[i_msd][2], sig_r_displ2[i_msd][2],
         xy_displ2[i_msd], sig_xy_displ2[i_msd],
         xyz_displ2[i_msd], sig_xyz_displ2[i_msd]);
   }
   fclose(f_output);

   /* Calculate diffusion constants from linear fits to the mean square
      displacements. */
   x = dvector(1, n_msd + 1);
   y = dvector(1, n_msd + 1);
   sig = dvector(1, n_msd + 1);
   n_fit = n_msd - first_msd + 1;
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = r_displ2[i_msd][0];
      sig[i_fit] = sig_r_displ2[i_msd][0];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_x = b / 2.0;
   sig_diffus_x = sigb / 2.0;
   f_output = fopen("pro.msd_fit_x", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = r_displ2[i_msd][1];
      sig[i_fit] = sig_r_displ2[i_msd][1];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_y = b / 2.0;
   sig_diffus_y = sigb / 2.0;
   f_output = fopen("pro.msd_fit_y", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = r_displ2[i_msd][2];
      sig[i_fit] = sig_z_displ2[i_msd][2];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_z = b / 2.0;
   sig_diffus_z = sigb / 2.0;
   f_output = fopen("pro.msd_fit_z", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = xy_displ2[i_msd];
      sig[i_fit] = sig_xy_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_xy = b / 4.0;
   sig_diffus_xy = sigb / 4.0;
   f_output = fopen("pro.msd_fit_xy", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      i_msd = i_fit + first_msd - 1;
      x[i_fit] = i_msd * interval;
      y[i_fit] = xyz_displ2[i_msd];
      sig[i_fit] = sig_xyz_displ2[i_msd];
   } 
   fit(x, y, n_fit, sig, 1, &a, &b, &siga, &sigb, &chi2, &q);
   diffus_xyz = b / 6.0;
   sig_diffus_xyz = sigb / 6.0;
   f_output = fopen("pro.msd_fit_xyz", "w");
   for (i_fit = 1; i_fit <= n_fit; ++i_fit) {
      y_fit = a + b * x[i_fit];
      fprintf(f_output, "%g %g %g %g\n", x[i_fit], y[i_fit], sig[i_fit], y_fit);
   } 
   fclose(f_output);

   /* Write diffusion constants to output file. */
   dum1 = 1.0e-4;
   dum2 = delta;
   f_output = fopen("pro.diffusion", "w");
   fprintf(f_output, "diffus_x = %g +/- %g angstrom^2 / ps\n",
      diffus_x, sig_diffus_x);
   fprintf(f_output, "         = %g +/- %g cm^2 / s\n",
      dum1 * diffus_x, dum1 * sig_diffus_x);
   fprintf(f_output, "         = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_x, dum2 * sig_diffus_x);
   fprintf(f_output, "diffus_y = %g +/- %g angstrom^2 / ps\n",
      diffus_y, sig_diffus_y);
   fprintf(f_output, "         = %g +/- %g cm^2 / s\n",
      dum1 * diffus_y, dum1 * sig_diffus_y);
   fprintf(f_output, "         = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_y, dum2 * sig_diffus_y);
   fprintf(f_output, "diffus_z = %g +/- %g angstrom^2 / ps\n",
      diffus_z, sig_diffus_z);
   fprintf(f_output, "         = %g +/- %g cm^2 / s\n",
      dum1 * diffus_z, dum1 * sig_diffus_z);
   fprintf(f_output, "         = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_z, dum2 * sig_diffus_z);
   fprintf(f_output, "diffus_xy = %g +/- %g angstrom^2 / ps\n",
      diffus_xy, sig_diffus_xy);
   fprintf(f_output, "          = %g +/- %g cm^2 / s\n",
      dum1 * diffus_xy, dum1 * sig_diffus_xy);
   fprintf(f_output, "          = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_xy, dum2 * sig_diffus_xy);
   fprintf(f_output, "diffus_xyz = %g +/- %g angstrom^2 / ps\n",
      diffus_xyz, sig_diffus_xyz);
   fprintf(f_output, "           = %g +/- %g cm^2 / s\n",
      dum1 * diffus_xyz, dum1 * sig_diffus_xyz);
   fprintf(f_output, "           = %g +/- %g angstrom^2 / MD step\n",
      dum2 * diffus_xyz, dum2 * sig_diffus_xyz);
   fclose(f_output);

   /* Free up memory. */
   free(mol_coords_corr);
   free(mol_coords_corr_0);
   free(r_displ2);
   free(sig_r_displ2);
   free(xy_displ2);
   free(sig_xy_displ2);
   free(xyz_displ2);
   free(sig_xyz_displ2);
   free(norm);
   free_dvector(x, 1, n_fit);
   free_dvector(y, 1, n_fit);
   free_dvector(sig, 1, n_fit);
}
