/* Initialization routines. */

#include "build.h"

/***************************************************************************/
/* Create an isotropic initial condition (molecules given random positions
   and orientations).

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int isotropic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int *graph_switch, int *graph_pore_switch, 
      int n_mols, int n_species, long *i_ran, int *mol_first_atm,
      int *n_atoms_per_mol, int *mol_species, int *atom_rel, int n_type,
      int *atom_type,
      double ***temp_atm_pos, double **h, double **h_inv, 
      double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, 
      double **rel_atom_coords, double **combine, int pore_switch, double z0, 
      double r0, int *n_bonds_per_mol, 
      int **temp_bonds_1, int **temp_bonds_2, int *species_choice,
      int *comb_col, int resph, double radius_sph, double *radius_cyl, 
      int x_cut_switch)
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, i_species, i, j;
   double norm;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp;
   short keep_going = 1;

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species and label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

         /* Generate scaled position. */
         for (j = 0; j < NDIM; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;

         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
            + h[0][1] * scaled_mol_coords[i_mol][1]
            + h[0][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
            + h[1][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         psi = TWO_PI * ran3(i_ran);
         cos_psi = cos(psi);
         sin_psi = sin(psi);
         cos_theta = 2.0 * ran3(i_ran) - 1.0;
         sin_theta = sqrt(1.0 - SQR(cos_theta));
         phi = TWO_PI * ran3(i_ran);
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule into standard coordinate systems. */
            /* Not used with an orthorombic box ! */
/*          x_tmp = x_hat.x * x_rot + y_hat.x * y_rot + z_hat.x * z_rot;
            y_tmp = x_hat.y * x_rot + y_hat.y * y_rot + z_hat.y * z_rot;
            z_tmp = x_hat.z * x_rot + y_hat.z * y_rot + z_hat.z * z_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;
            z_rot = z_tmp;
*/
            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

         /* Check for overlaps. */
         overlap_flag = 0;

         /* If a pore is present, check also overlap of the first molecule
            with the pore walls. */
         if ((pore_switch) && (i_mol == 0)){
            if (pore(i_mol, mol_first_atm, n_atoms_per_mol, mol_species,
                     n_type, atom_type, combine, h, mol_coords, 
                     scaled_atom_coords, rel_atom_coords, z0, r0))
                 overlap_flag = 1;
         }

         for (j_mol = 0; j_mol < i_mol; ++j_mol)
         {
            if (overlaps(i_mol, j_mol, mol_first_atm, n_atoms_per_mol, 
                  mol_species, atom_rel, n_type, atom_type, h, 
                  scaled_atom_coords, 
                  combine, mol_coords, rel_atom_coords, pore_switch, z0, r0))
            {
               overlap_flag = 1;
               break;
            }
         }
             
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */
#ifdef GRAPHICS
if (*graph_switch > 0)
  if ((*species_choice == -1)
              || (*species_choice > -1 && i_species == *species_choice)){
      redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1, mol_species,
             mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
             mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2,
             n_species, species_choice, atom_type, comb_col, resph, radius_sph,
             radius_cyl, &keep_going, x_cut_switch, pore_switch);
       if (pore_switch)
       redraw_pore(dpy, win, doubleBuffer, graph_pore_switch, resph, z0, r0, h,
                   x_cut_switch, comb_col, n_type);
  }
#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create a nematic initial condition (molecules given random positions
   and oriented along z axis).

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int nematic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int *graph_switch, int *graph_pore_switch, 
      int n_mols, int n_species, long *i_ran, int *mol_first_atm,
      int *n_atoms_per_mol, int *mol_species, int *atom_rel, int n_type,
      int *atom_type,
      double ***temp_atm_pos, double **h, double **h_inv, 
      double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, 
      double **rel_atom_coords, double **combine, int pore_switch,
      double z0, double r0, int *n_bonds_per_mol, 
      int **temp_bonds_1, int **temp_bonds_2, int *species_choice,
      int *comb_col, int resph, double radius_sph, double *radius_cyl, 
      int x_cut_switch)
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, i_species, i, j;
   double norm, signe;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp;
   short keep_going = 1;

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species and label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
          }

         /* Generate scaled position. */
         for (j = 0; j < NDIM; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;
         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
            + h[0][1] * scaled_mol_coords[i_mol][1]
            + h[0][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
            + h[1][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         psi = TWO_PI * ran3(i_ran);
         cos_psi = cos(psi);
         sin_psi = sin(psi);
	 signe = ran3(i_ran) - 0.5;
         if (signe == 0.0) signe = 1.0;
         else signe = signe / fabs(signe);
         cos_theta = signe;
         sin_theta = 0.0;
         phi = TWO_PI * ran3(i_ran);
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule into standard coordinate systems. */
            /* Not used with an orthorombic box ! */
/*          x_tmp = x_hat.x * x_rot + y_hat.x * y_rot + z_hat.x * z_rot;
            y_tmp = x_hat.y * x_rot + y_hat.y * y_rot + z_hat.y * z_rot;
            z_tmp = x_hat.z * x_rot + y_hat.z * y_rot + z_hat.z * z_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;
            z_rot = z_tmp;
*/
            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

         /* Check for overlaps. */
         overlap_flag = 0;

         /* If a pore is present, check also overlap of the first molecule
            with the pore walls. */
         if ((pore_switch) && (i_mol == 0))
            if (pore(i_mol, mol_first_atm, n_atoms_per_mol, mol_species,
                     n_type, atom_type, combine, h, mol_coords, 
                     scaled_atom_coords, rel_atom_coords, z0, r0))
                 overlap_flag = 1;

         for (j_mol = 0; j_mol < i_mol; ++j_mol)
         {
            if (overlaps(i_mol, j_mol, mol_first_atm, n_atoms_per_mol, 
                  mol_species, atom_rel, n_type, atom_type, h, 
                  scaled_atom_coords, 
                  combine, mol_coords, rel_atom_coords, pore_switch, z0, r0))
            {
               overlap_flag = 1;
               break;
            }
         }
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */
#ifdef GRAPHICS
if (*graph_switch > 0)
  if ((*species_choice == -1)
              || (*species_choice > -1 && i_species == *species_choice)){
      redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1, mol_species,
             mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
             mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2,
             n_species, species_choice, atom_type, comb_col, resph, radius_sph,
             radius_cyl, &keep_going, x_cut_switch, pore_switch);
       if (pore_switch)
         redraw_pore(dpy, win, doubleBuffer, graph_pore_switch, resph, z0, r0, 
                     h, x_cut_switch, comb_col, n_type);
  }
#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create a smectic initial condition (molecules arranged in smectic layers
   with random positions within the layer planes and with the principal axe
   of all molecules oriented along the z-axis).*/
/* 
input: 
       Number of molecules (n_mols)
       Number of molecular species (n_species)
       Number of smectic layers (n_layers)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int smectic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int *graph_switch, int *graph_pore_switch, 
      int n_mols, int n_species, long *i_ran, int n_layers, 
      int *n_mols_per_species, int *mol_first_atm,
      int *n_atoms_per_mol, int *mol_species, int *atom_rel, int n_type,
      int *atom_type,
      double ***temp_atm_pos, double **h, double **h_inv, 
      double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, 
      double **rel_atom_coords, double **combine, int pore_switch,
      double z0, double r0, int *n_bonds_per_mol, 
      int **temp_bonds_1, int **temp_bonds_2, int *species_choice,
      int *comb_col, int resph, double radius_sph, double *radius_cyl, 
      int x_cut_switch)
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, i_species, i, j,
      **layer_counter, *n_per_layer, i_layer, layer_tmp;
   double norm, scaled_layer_spacing, signe;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp;
   short keep_going = 1;

  /* Allocate memory for temporary arrays. */
  layer_counter = allocate_2d_array(n_species, n_layers, sizeof(int));
  n_per_layer = allocate_1d_array(n_species, sizeof(int));

   /* Loop over molecular species. */
   for (i_species = 0; i_species < n_species; ++i_species) {
       layer_tmp = n_mols_per_species[i_species] % n_layers;
       if (layer_tmp == 0)
          n_per_layer[i_species] = n_mols_per_species[i_species] / 
                                                         (double)n_layers; 
       else
	  error_exit("Wrong number of mol per layer");	
   }
   scaled_layer_spacing = 1.0 / n_layers;

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species and label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in smectic_start\n");
            return(1);
          }

          /* Pick a layer at random. */
          do {
             i_layer = (int) (n_layers * ran3(i_ran));
             if (i_layer == n_layers) i_layer = 0.0;
         } while (layer_counter[i_species][i_layer] == n_per_layer[i_species]);


         /* Generate scaled position. */
         for (j = 0; j < NDIM - 1; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;
         scaled_mol_coords[i_mol][2] = (i_layer + 0.5) * scaled_layer_spacing
                                       - 0.5;
 
         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
            + h[0][1] * scaled_mol_coords[i_mol][1]
            + h[0][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
            + h[1][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation (principal molecular axis along z). */
         psi = TWO_PI * ran3(i_ran);
         cos_psi = cos(psi);
         sin_psi = sin(psi);
         signe = ran3(i_ran) - 0.5;
         if (signe == 0.0) signe = 1.0;
         else signe = signe / fabs(signe); 
         cos_theta = signe;
         sin_theta = 0.0;
         phi = TWO_PI * ran3(i_ran);
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule into standard coordinate systems. */
            /* Not used with an orthorombic box ! */
/*          x_tmp = x_hat.x * x_rot + y_hat.x * y_rot + z_hat.x * z_rot;
            y_tmp = x_hat.y * x_rot + y_hat.y * y_rot + z_hat.y * z_rot;
            z_tmp = x_hat.z * x_rot + y_hat.z * y_rot + z_hat.z * z_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;
            z_rot = z_tmp;
*/
            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

         /* Check for overlaps. */
         overlap_flag = 0;

         /* If a pore is present, check also overlap of the first molecule
            with the pore walls. */
         if ((pore_switch) && (i_mol == 0))
            if (pore(i_mol, mol_first_atm, n_atoms_per_mol, mol_species,
                     n_type, atom_type, combine, h, mol_coords, 
                     scaled_atom_coords, rel_atom_coords, z0, r0))
                 overlap_flag = 1;

         for (j_mol = 0; j_mol < i_mol; ++j_mol)
         {
            if (overlaps(i_mol, j_mol, mol_first_atm, n_atoms_per_mol, 
                  mol_species, atom_rel, n_type, atom_type, h, 
                  scaled_atom_coords, 
                  combine, mol_coords, rel_atom_coords, pore_switch, z0, r0))
            {
               overlap_flag = 1;
               break;
            }
         }
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */
#ifdef GRAPHICS
if (*graph_switch > 0)
  if ((*species_choice == -1)
              || (*species_choice > -1 && i_species == *species_choice)){
      redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1, mol_species,
             mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
             mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2,
             n_species, species_choice, atom_type, comb_col, resph, radius_sph,
             radius_cyl, &keep_going, x_cut_switch, pore_switch);
       if (pore_switch)
         redraw_pore(dpy, win, doubleBuffer, graph_pore_switch, resph, z0, r0, 
                     h, x_cut_switch, comb_col, n_type);
  }
#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}
/***************************************************************************/
/* Create a single chain in the middle of the pore. Chain is parallel to 
   the cylinder axis.

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int single_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int *graph_switch, int *graph_pore_switch, 
      int n_mols, int n_species, long *i_ran, int *mol_first_atm,
      int *n_atoms_per_mol, int *mol_species, int *n_mols_per_species,
      int *atom_rel, int n_type, int *atom_type,
      double ***temp_atm_pos, double **h, double **h_inv, 
      double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, 
      double **rel_atom_coords, double **combine, int pore_switch,
      double z0, double r0, int *n_bonds_per_mol, 
      int **temp_bonds_1, int **temp_bonds_2, int *species_choice,
      int *comb_col, int resph, double radius_sph, double *radius_cyl, 
      int x_cut_switch)
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, i_species, 
      single_specie, i, j;
   double norm, signe;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp;
   short keep_going = 1;

   /* Find which specie is the single chain molecule. */
   for (i_species = 0; i_species < n_species; ++i_species){
       if (n_mols_per_species[i_species] == 1){
          single_specie = i_species;
          break;
       }
   }

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species and label of first atom in molecule. */
      i_species = mol_species[i_mol];
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
          }

         /* Generate scaled position. */
         for (j = 0; j < NDIM; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;

         /* If i_specie = single_specie, try to insert in the middle of
            the pore and exit if not successful. */
         /* we assume the pore is centered around the origin. */
         if (i_species == single_specie){
            scaled_mol_coords[i_mol][0] = 0.0;
            scaled_mol_coords[i_mol][1] = 0.0;
            scaled_mol_coords[i_mol][2] = 0.0;
            }
               
         /* Molecular position. */
         mol_coords[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0]
            + h[0][1] * scaled_mol_coords[i_mol][1]
            + h[0][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1]
            + h[1][2] * scaled_mol_coords[i_mol][2];
         mol_coords[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         psi = TWO_PI * ran3(i_ran);
         cos_psi = cos(psi);
         sin_psi = sin(psi);
         cos_theta = 2.0 * ran3(i_ran) - 1.0;
         sin_theta = sqrt(1.0 - SQR(cos_theta));
         phi = TWO_PI * ran3(i_ran);
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* align the single chain along the cylinder axis.*/
         if (i_species == single_specie){
	    signe = ran3(i_ran) - 0.5;
            if (signe == 0.0) signe = 1.0;
            else signe = signe / fabs(signe);
            cos_theta = signe;
            sin_theta = 0.0;
         }

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i_species][i][0];
            y_rot = temp_atm_pos[i_species][i][1];
            z_rot = temp_atm_pos[i_species][i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule into standard coordinate systems. */
            /* Not used with an orthorombic box ! */
/*          x_tmp = x_hat.x * x_rot + y_hat.x * y_rot + z_hat.x * z_rot;
            y_tmp = x_hat.y * x_rot + y_hat.y * y_rot + z_hat.y * z_rot;
            z_tmp = x_hat.z * x_rot + y_hat.z * y_rot + z_hat.z * z_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;
            z_rot = z_tmp;
*/
            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
             mol_species, h_inv, atom_coords, scaled_atom_coords);

         /* Check for overlaps. */
         overlap_flag = 0;

         /* If a pore is present, check also overlap of the first molecule
            with the pore walls. */
         if ((pore_switch) && (i_mol == 0))
            if (pore(i_mol, mol_first_atm, n_atoms_per_mol, mol_species,
                     n_type, atom_type, combine, h, mol_coords, 
                     scaled_atom_coords, rel_atom_coords, z0, r0))
                 overlap_flag = 1;

         for (j_mol = 0; j_mol < i_mol; ++j_mol)
         {
            if (overlaps(i_mol, j_mol, mol_first_atm, n_atoms_per_mol, 
                  mol_species, atom_rel, n_type, atom_type, h, 
                  scaled_atom_coords, 
                  combine, mol_coords, rel_atom_coords, pore_switch, z0, r0))
            {
               overlap_flag = 1;
               break;
            }
         }

         /* If it is the single chain molecule, and if an overlap is detected,
            exit. */
         if ((i_species == single_specie) && (overlap_flag == 1))
            error_exit("Either the diameter of the pore is too small to accomodate the chain, either the single chain wasn't inserted first");
             
            
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */
#ifdef GRAPHICS
if (*graph_switch > 0)
  if ((*species_choice == -1)
              || (*species_choice > -1 && i_species == *species_choice)){
      redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1, mol_species,
             mol_first_atm, n_atoms_per_mol, n_bonds_per_mol,
             mol_coords, rel_atom_coords, h, temp_bonds_1, temp_bonds_2,
             n_species, species_choice, atom_type, comb_col, resph, radius_sph,
             radius_cyl, &keep_going, x_cut_switch, pore_switch);
       if (pore_switch)
         redraw_pore(dpy, win, doubleBuffer, graph_pore_switch, resph, z0, r0, 
                     h, x_cut_switch, comb_col, n_type);
  }
#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/****************************************************************************/
/* Check for overlaps between molecule i_mol and molecule j_mol,
   with periodic boundary conditions, using N^2 search */
/*
input :
      Labels of the pair of moelcules (i_mol, j_mol)
      Array of the label of first atom in molecules (mol_first_atom)
      Array of the number of atoms per template molecule (n_atoms_per_mol)
      Array of the label of molecular species (mol_species)
      Array of the realtive label - in the template mol - of atoms (atom_rel) 
      Array of the orthorombic box matrix (h)
      Array of the scaled atomic positions (scaled_atom_coords)
      Array of the overlaps criterion between 2 atomic sites (combine)
output :
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int overlaps(int i_mol, int j_mol, int *mol_first_atm, 
      int *n_atoms_per_mol, int *mol_species, int *atom_rel, int n_type,
      int *atom_type,
      double **h, double **scaled_atom_coords, double **combine,
      double **mol_coords, double **rel_atom_coords, int pore_switch, 
      double z0, double r0)
{
   int overlap, skip_i, skip_j, i, j, k, i_species, j_species,
      i_rel, j_rel, i_type, j_type;
   double s_sep[NDIM], sep[NDIM], r2_sep, s_i[NDIM]; 

   /* Get label of molecular species and label of first atom in molecule. */
   i_species = mol_species[i_mol];
   skip_i = mol_first_atm[i_mol];

   /* Loop over atoms in molecule i_mol. */
   for (i = skip_i; i < skip_i + n_atoms_per_mol[i_species]; ++i) {

      /* Get attributes of atom i. */
      for (k = 0; k < NDIM; ++k)
        s_i[k] = scaled_atom_coords[i][k];
      i_rel = atom_rel[i];
      i_type = atom_type[i];

      /* Get label of molecular species and label of first atom in molecule. */
      j_species = mol_species[j_mol];
      skip_j = mol_first_atm[j_mol];

      /* Loop over atoms in molecule j_mol. */
      for (j = skip_j; j < skip_j + n_atoms_per_mol[j_species]; ++j) {

           /* Calculate scaled pair separation. */
           for (k = 0; k < NDIM; ++k)
              {
              s_sep[k] = scaled_atom_coords[j][k] - s_i[k];
              /* Apply periodic boundary conditions */
              s_sep[k] -= NINT(s_sep[k]);
              }
           sep[0] = h[0][0] * s_sep[0] + h[0][1] * s_sep[1] 
                    + h[0][2] * s_sep[2];
           sep[1] = h[1][1] * s_sep[1] + h[1][2] * s_sep[2];
           sep[2] = h[2][2] * s_sep[2];
           r2_sep = SQR(sep[0]) + SQR(sep[1]) + SQR(sep[2]);

           /* If an overlap is detected, exit returning 1 */ 
           j_rel = atom_rel[j];
           j_type = atom_type[j];

           if (r2_sep < combine[i_type][j_type]) 
                  return(1);
      }
   }

   /* If we have a pore, check if the molecule i_mol is outside the walls,
      and if yes, return 1. */
   if (pore_switch)
      if (pore(i_mol, mol_first_atm, n_atoms_per_mol, mol_species,
               n_type, atom_type, combine,
               h, mol_coords, scaled_atom_coords, rel_atom_coords, z0, r0))
          return(1);

   /* Return 0, ie no overlap found between molecules mol_i and mol_j. */
   return(0);
}
