
/* Initialization routines. */

#include "shared.h"

#define MAX_TRIALS	1000000
/***************************************************************************/
/* Create an isotropic initial condition (molecules given random positions
   and orientations).

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/
int isotropic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, int n_mol, double length, 
      double skin, double *side, 
      double *side_inv, long *i_ran, int *mol_first_atm, int n_atoms_per_mol, 
      double **temp_atm_pos,  double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, int n_sphero_per_mol,
      int *temp_bonds_1, int *temp_bonds_2, int resph, double radius_sph,
      double radius_cyl) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, **rel_atom_coords;
   short keep_going = 1;

   /* allocate memory for 'temporary' relative atomic coordinates. */
   rel_atom_coords = allocate_2d_array(3 * n_mol, 3, sizeof(double));

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mol; ++i_mol) {

      /* Get label of first atom in molecule. */
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

         /* Generate scaled position. */
         for (j = 0; j < 3; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;

         /* Molecular position. */
         mol_coords[i_mol][0] = side[0] * scaled_mol_coords[i_mol][0];
         mol_coords[i_mol][1] = side[1] * scaled_mol_coords[i_mol][1];
         mol_coords[i_mol][2] = side[2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         psi = TWO_PI * ran3(i_ran);
         cos_psi = cos(psi);
         sin_psi = sin(psi);
         cos_theta = 2.0 * ran3(i_ran) - 1.0;
         sin_theta = sqrt(1.0 - SQR(cos_theta));
         phi = TWO_PI * ran3(i_ran);
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i][0];
            y_rot = temp_atm_pos[i][1];
            z_rot = temp_atm_pos[i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single_old(i_mol, n_atoms_per_mol, mol_first_atm,
             side_inv, atom_coords, scaled_atom_coords);

         norm1 = norm2 = 0.0;
         for (k = 0; k < 3; ++k) {
            s[0][k] = scaled_atom_coords[skip + 1][k] -  
                      scaled_atom_coords[skip][k];  
            s[0][k] -= NINT(s[0][k]);
            u[0][k] = s[0][k] * side[k] / length;
            s[0][k] = s[0][k] * 0.5 + scaled_atom_coords[skip][k];
            s[0][k] -= NINT(s[0][k]);

            s[1][k] = scaled_atom_coords[skip + 2][k] -  
                      scaled_atom_coords[skip][k];  
            s[1][k] -= NINT(s[1][k]);
            u[1][k] = s[1][k] * side[k] / length;
            s[1][k] = s[1][k] * 0.5 + scaled_atom_coords[skip][k];
            s[1][k] -= NINT(s[1][k]);

            norm1 += u[0][k] * u[0][k];
            norm2 += u[1][k] * u[1][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-05) || (fabs(norm2 - 1.0) > 1e-05)){
            printf("pb with norm.");
            exit(1);
            }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           
           for (i = 0; i < 2; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
               u1[k] = s1[k] * side[k] / length;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-05){
                printf("pb with norm.");
                exit(1);
             }
             for (j = 0; j < 2; ++j)
             if (overlap(s[j], u[j], s1, u1, length, 0.0, side)) {
                 overlap_flag = 1;
                 break;
                 }
             if (overlap_flag) 
               break;
           }
           if (overlap_flag) 
             break;
         }
         /* Increment trial counter. */
         ++n_trials;
      } while (overlap_flag);

      /* Plot molecular configuration. */
#ifdef GRAPHICS
  if (graph_switch > 0)
     redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_first_atm, n_atoms_per_mol, n_sphero_per_mol,
                 atom_coords, scaled_atom_coords, mol_coords,
                 rel_atom_coords, side, temp_bonds_1, temp_bonds_2, resph,
                 radius_sph, radius_cyl, &keep_going);


#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mol);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create an nematic initial condition (molecules given random positions
   and orientations).

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int nematic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, int polar_switch,
      int n_mol, double length, double skin, double *side, 
      double *side_inv, long *i_ran, int *mol_first_atm, int n_atoms_per_mol, 
      double **temp_atm_pos,  double **mol_coords, double **scaled_mol_coords,
      double **atom_coords, double **scaled_atom_coords, int n_sphero_per_mol,
      int *temp_bonds_1, int *temp_bonds_2, int resph, double radius_sph, 
      double radius_cyl) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, signe, **rel_atom_coords;
   short keep_going = 1;

printf("side : %g %g %g\n", side[0], side[1], side[2]);
fflush(NULL);
   
   /* allocate memory for 'temporary' relative atomic coordinates. */
   rel_atom_coords = allocate_2d_array(3 * n_mol, 3, sizeof(double));

printf("polar_switch = %d\n", polar_switch);
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mol; ++i_mol) {

      /* Get label of first atom in molecule. */
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

         /* Generate scaled position. */
         for (j = 0; j < 3; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;

         /* Molecular position. */
         mol_coords[i_mol][0] = side[0] * scaled_mol_coords[i_mol][0];
         mol_coords[i_mol][1] = side[1] * scaled_mol_coords[i_mol][1];
         mol_coords[i_mol][2] = side[2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         if (polar_switch) {
           psi = 0.0; 
           psi = 0.0; 
         }
         else {
           psi = TWO_PI * ran3(i_ran); 
           phi = TWO_PI * ran3(i_ran); 
         }

         cos_psi = cos(psi);
         sin_psi = sin(psi);
/*         signe = ran3(i_ran) - 0.5;
         if (signe == 0.0) signe = 1.0;
         else signe = signe / fabs(signe);  
*/
         signe = 1.0;
         cos_theta = signe;
         sin_theta = 0.0;
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i][0];
            y_rot = temp_atm_pos[i][1];
            z_rot = temp_atm_pos[i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single_old(i_mol, n_atoms_per_mol, mol_first_atm,
             side_inv, atom_coords, scaled_atom_coords);

         norm1 = norm2 = 0.0;
         for (k = 0; k < 3; ++k) {
            s[0][k] = scaled_atom_coords[skip + 1][k] -  
                      scaled_atom_coords[skip][k];  
            s[0][k] -= NINT(s[0][k]);
            u[0][k] = s[0][k] * side[k] / length;
            s[0][k] = s[0][k] * 0.5 + scaled_atom_coords[skip][k];
            s[0][k] -= NINT(s[0][k]);

            s[1][k] = scaled_atom_coords[skip + 2][k] -  
                      scaled_atom_coords[skip][k];  
            s[1][k] -= NINT(s[1][k]);
            u[1][k] = s[1][k] * side[k] / length;
            s[1][k] = s[1][k] * 0.5 + scaled_atom_coords[skip][k];
            s[1][k] -= NINT(s[1][k]);

            norm1 += u[0][k] * u[0][k];
            norm2 += u[1][k] * u[1][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-05) || (fabs(norm2 - 1.0) > 1e-05)){
            printf("pb with norm.");
            exit(1);
            }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           
           for (i = 0; i < 2; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
               u1[k] = s1[k] * side[k] / length;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-05){
                printf("pb with norm.");
                exit(1);
             }
             for (j = 0; j < 2; ++j)
             if (overlap(s[j], u[j], s1, u1, length, 0.0, side)) {
                 overlap_flag = 1;
                 break;
                 }
             if (overlap_flag) 
               break;
           }
           if (overlap_flag) 
             break;
         }
/*
if (overlap_flag){
   printf("i_mol = %d, j_mol = %d\n", i_mol, j_mol);
printf("  sx = %g, sy = %g, sz = %g\n", scaled_mol_coords[i_mol][0], scaled_mol_coords[i_mol][1], scaled_mol_coords[i_mol][2]); 
exit(1);
   }
*/
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Plot molecular configuration. */

#ifdef GRAPHICS
  if (graph_switch > 0)
          redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_first_atm, n_atoms_per_mol, n_sphero_per_mol,
                 atom_coords, scaled_atom_coords, mol_coords,
                 rel_atom_coords, side, temp_bonds_1, temp_bonds_2, resph,
                 radius_sph, radius_cyl, &keep_going);

#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
/*printf("  sx = %g, sy = %g, sz = %g\n", scaled_mol_coords[i_mol][0], scaled_mol_coords[i_mol][1], scaled_mol_coords[i_mol][2]); 
      fflush(NULL);
*/
   }

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mol);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create a smectic initial condition (molecules given random positions
   and orientations).

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int smectic_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, int polar_switch, int n_mol, 
      int n_layers, 
      double length, double skin, 
      double *side, double *side_inv, long *i_ran, int *mol_first_atm, 
      int n_atoms_per_mol, double **temp_atm_pos,  double **mol_coords, 
      double **scaled_mol_coords, double **atom_coords, 
      double **scaled_atom_coords, int n_sphero_per_mol,
      int *temp_bonds_1, int *temp_bonds_2, int resph, double radius_sph, 
      double radius_cyl) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k,
      *layer_counter, n_per_layer, i_layer, i_number, layer_tmp;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, signe, scaled_layer_spacing,
      **rel_atom_coords;
   short keep_going = 1;
  
   /* Allocate memory for temporary array. */
   layer_counter = allocate_1d_array(n_layers, sizeof(int));
   rel_atom_coords = allocate_2d_array(3 * n_mol, 3, sizeof(double));

   layer_tmp = n_mol % n_layers;
     if (layer_tmp == 0)
       n_per_layer = n_mol / n_layers;
     else{
       printf("Wrong number of mol per layer");
       exit(1);
       }
   scaled_layer_spacing = 1.0 / n_layers;

   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mol; ++i_mol) {

      /* Get label of first atom in molecule. */
      skip = mol_first_atm[i_mol];

      /* Generate trial positions and orientations for molecule i_mol 
         until an overlap-free set is found. */
      n_trials = 0;
      do {

         /* Exit with nonzero return value if maximum number of trial
            insertions is exceeded. */
         if (n_trials > MAX_TRIALS) {
            printf("Maximum number of trials exceeded in isotropic_start\n");
            return(1);
            }

          /* Pick a layer at random. */
          do {
             i_layer = (int) (n_layers * ran3(i_ran));
             i_number = (i_layer % 2 == 0);
             if (i_layer == n_layers) i_layer = 0.0;
         } while (layer_counter[i_layer] == n_per_layer);

         /* Generate scaled position. */
         for (j = 0; j < 2; ++j)
           scaled_mol_coords[i_mol][j] = ran3(i_ran) - 0.5;
         scaled_mol_coords[i_mol][2] = (i_layer + 0.5) * scaled_layer_spacing
                                       - 0.5;

         /* Molecular position. */
         mol_coords[i_mol][0] = side[0] * scaled_mol_coords[i_mol][0];
         mol_coords[i_mol][1] = side[1] * scaled_mol_coords[i_mol][1];
         mol_coords[i_mol][2] = side[2] * scaled_mol_coords[i_mol][2];

         /* Choose molecular orientation at random. */
         /* If polar_switch == 2, built a smectic antipolar order.
            If polar_switch == 1, built a smectic polar order.
            If polar_switch == 0, built a smectic azimutally isotropic order. */

         if (polar_switch == 2)
            if (i_number) {
               psi = 0.0;
               phi = 0.0;
            }
            else {
               psi = 0.0;
               phi = PI;
            }
         else if (polar_switch == 1) {
            psi = 0.0;
            phi = 0.0;
         }
         else{ 
            psi = TWO_PI * ran3(i_ran);
            phi = TWO_PI * ran3(i_ran);
         }


         cos_psi = cos(psi);
         sin_psi = sin(psi);
     /*    signe = ran3(i_ran) - 0.5;
         if (signe == 0.0) signe = 1.0;
         else signe = signe / fabs(signe);
*/
         signe = 1.0;
         cos_theta = signe;
         sin_theta = 0.0;
         cos_phi = cos(phi);
         sin_phi = sin(phi);

         /* Calculate atomic positions. The rotations are performed
            in the unit-cell-based reference systems, and then
            coordinates are transformed into the "standard"
            coordinate system. */
         for (i = 0; i < n_atoms_per_mol; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i][0];
            y_rot = temp_atm_pos[i][1];
            z_rot = temp_atm_pos[i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
         }

         /* Calculate scaled atomic coordinates for molecule i_mol. */
         scaled_atomic_coords_single_old(i_mol, n_atoms_per_mol, mol_first_atm,
             side_inv, atom_coords, scaled_atom_coords);

         norm1 = norm2 = 0.0;
         for (k = 0; k < 3; ++k) {
            s[0][k] = scaled_atom_coords[skip + 1][k] -  
                      scaled_atom_coords[skip][k];  
            s[0][k] -= NINT(s[0][k]);
            u[0][k] = s[0][k] * side[k] / length;
            s[0][k] = s[0][k] * 0.5 + scaled_atom_coords[skip][k];
            s[0][k] -= NINT(s[0][k]);

            s[1][k] = scaled_atom_coords[skip + 2][k] -  
                      scaled_atom_coords[skip][k];  
            s[1][k] -= NINT(s[1][k]);
            u[1][k] = s[1][k] * side[k] / length;
            s[1][k] = s[1][k] * 0.5 + scaled_atom_coords[skip][k];
            s[1][k] -= NINT(s[1][k]);

            norm1 += u[0][k] * u[0][k];
            norm2 += u[1][k] * u[1][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-05) || (fabs(norm2 - 1.0) > 1e-05)){
            printf("pb with norm.");
            exit(1);
            }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           
           for (i = 0; i < 2; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
               u1[k] = s1[k] * side[k] / length;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-05){
                printf("pb with norm.");
                exit(1);
             }
             for (j = 0; j < 2; ++j)
             if (overlap(s[j], u[j], s1, u1, length, 0.0, side)) {
                 overlap_flag = 1;
                 break;
                 }
             if (overlap_flag) 
               break;
           }
           if (overlap_flag) 
             break;
         }
         /* Increment trial counter. */
         ++n_trials;

      } while (overlap_flag);

      /* Increment counter. */
      ++layer_counter[i_layer];

      /* Plot molecular configuration. */

#ifdef GRAPHICS
  if (graph_switch > 0)
          redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_first_atm, n_atoms_per_mol, n_sphero_per_mol,
                 atom_coords, scaled_atom_coords, mol_coords,
                 rel_atom_coords, side, temp_bonds_1, temp_bonds_2, resph,
                 radius_sph, radius_cyl, &keep_going);

#endif

      /* Print out number of trials required for successful 
         particle insertion. */
      printf("i_mol = %d, n_trials = %d\n", i_mol, n_trials);
      fflush(NULL);
   }

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mol);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
/* Create a crystal initial condition.

input: 
       Number of molecules (n_mols)
       Random number generator seed (i_ran)
       Array of label of first atom in template molecule (mol_first_atm)
       Array of number of atoms per molecule (n_atoms_per_mol)
       Array of label of molecular species (mol_species) 
       Array of relative label - within template mol - of atoms (atom_rel)
       Array of template atom - relative position of atomic sites (temp_atm_pos)
       Array of orthorombic box dimensions (h)
       Array of inverse box dimensions (h_inv)
       Array of overlap criterion between 2 atomic sites (combine)
       Array of number of bonds per template molecule (n_bonds_per_mol)
       Arrays of template molecule - labels of sites sharing a bond
                                                       (temp_bonds_1,
                                                        temp_bonds_2)
output: 
        Array of molecular positions (mol_coords)
        Array of scaled molecular positions (scaled_mol_coords)
        Array of atomic coordinates (atom_coords)
        Array of scaled atomic coordinates (scaled_atom_coords) 
        Flag indicating whether initial condition was successfully constructed;
        0 if so, 1 if not (return value) 
*/

int crystal_start(Display *dpy, Window win, GLboolean doubleBuffer,
      GLUquadricObj *qobj, int graph_switch, int polar_switch, int n_mol, 
      double length, int nc_x, int nc_y, int nc_z,
      double *a1, double *b1, double *c1, double dx, double dy, double skin, 
      double *side, double *side_inv, long *i_ran, int *mol_first_atm, 
      int n_atoms_per_mol, double **temp_atm_pos,  double **mol_coords, 
      double **scaled_mol_coords, double **atom_coords, 
      double **scaled_atom_coords, int n_sphero_per_mol,
      int *temp_bonds_1, int *temp_bonds_2, int resph, double radius_sph, 
      double radius_cyl) 
{
   int i_mol, j_mol, n_trials, overlap_flag, skip, abs, skip_j, i, j, k,
      *layer_counter, n_per_layer, i_layer, i_number, layer_tmp, nx, ny,
      lx, ly, lz;
   double norm, s[2][3], u[2][3], s1[3], u1[3], norm1, norm2;
   double psi, cos_psi, sin_psi, cos_theta, sin_theta, phi, cos_phi, sin_phi,
      x_rot, y_rot, z_rot, x_tmp, y_tmp, z_tmp, signe, scaled_layer_spacing,
      **rel_atom_coords, x, y, z, x0, y0, z0, shift_x;
   short keep_going = 1;
 
   /* Allocate memory for temporary array. */
   rel_atom_coords = allocate_2d_array(3 * n_mol, 3, sizeof(double));

   /* Initial position of the unit cell. */
   x0 = - side[0] / 2.0;
   y0 = - side[1] / 2.0;
   z0 = - side[2] / 2.0;

   /* Initialize molecular counter. */
   i_mol = 0;

   /* Loop over the lattice vectors. */
   for (lz = 0; lz < nc_z; ++lz) {
     z = z0 + lz * c1[2];
     for (ly = 0; ly < nc_y; ++ly) {
       shift_x = (ly % 2) * b1[0];
       y = y0 + ly * b1[1];
       for (lx = 0; lx < nc_x; ++lx) {
         x = x0 + lx * a1[0] + shift_x;

         /* Set position of the first molecule in the given unit cell. */
         mol_coords[i_mol][0] = x + a1[0] / 4.0; 
         mol_coords[i_mol][1] = y + b1[1] / 4.0; 
         mol_coords[i_mol][2] = z + c1[2] / 4.0;
         
	 /* Generate scaled position. */
         for (j = 0; j < 3; ++j){
           scaled_mol_coords[i_mol][j] = mol_coords[i_mol][j] / side[j];
           scaled_mol_coords[i_mol][j] -= NINT(scaled_mol_coords[i_mol][j]);
           mol_coords[i_mol][j] = scaled_mol_coords[i_mol][j] * side[j];
           }

           /* Choose molecular orientation at random. */
           /* If polar_switch == 2, built a crystal antipolar order.
              If polar_switch == 1, built a crystal polar order. */
           psi = 0.0;
           phi = PI / 2.0;
           cos_psi = cos(psi);
           sin_psi = sin(psi);
           signe = 1.0;
           cos_theta = signe;
           sin_theta = 0.0;
           cos_phi = cos(phi);
           sin_phi = sin(phi);

           /* Get label of first atom in molecule. */
           skip = mol_first_atm[i_mol];
 
           /* Calculate atomic positions. The rotations are performed
              in the unit-cell-based reference systems, and then
              coordinates are transformed into the "standard"
               coordinate system. */
           for (i = 0; i < n_atoms_per_mol; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i][0];
            y_rot = temp_atm_pos[i][1];
            z_rot = temp_atm_pos[i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
          }

          /* Calculate scaled atomic coordinates for molecule i_mol. */
          scaled_atomic_coords_single_old(i_mol, n_atoms_per_mol, mol_first_atm,
             side_inv, atom_coords, scaled_atom_coords);

          norm1 = norm2 = 0.0;
          for (k = 0; k < 3; ++k) {
            s[0][k] = scaled_atom_coords[skip + 1][k] -  
                      scaled_atom_coords[skip][k];  
            s[0][k] -= NINT(s[0][k]);
            u[0][k] = s[0][k] * side[k] / length;
            s[0][k] = s[0][k] * 0.5 + scaled_atom_coords[skip][k];
            s[0][k] -= NINT(s[0][k]);

            s[1][k] = scaled_atom_coords[skip + 2][k] -  
                      scaled_atom_coords[skip][k];  
            s[1][k] -= NINT(s[1][k]);
            u[1][k] = s[1][k] * side[k] / length;
            s[1][k] = s[1][k] * 0.5 + scaled_atom_coords[skip][k];
            s[1][k] -= NINT(s[1][k]);

            norm1 += u[0][k] * u[0][k];
            norm2 += u[1][k] * u[1][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-05) || (fabs(norm2 - 1.0) > 1e-05)){
            printf("pb with norm: norm1 = %g, norm2 = %g", norm1, norm2);
            exit(1);
            }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           
           for (i = 0; i < 2; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
               u1[k] = s1[k] * side[k] / length;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-05){
                printf("pb with norm.");
                exit(1);
             }
             for (j = 0; j < 2; ++j)
             if (overlap(s[j], u[j], s1, u1, length, 0.0, side)) {
                 printf("overlap in the close-packed crystal structure.\n");
                 printf("i_mol = %d, j_mol = %d\n", i_mol, j_mol);
                 exit(1);  
                 }
           }
         }

          /* Handle the second molecule. */
          ++i_mol;

          /* If polar_switch == 2, built a crystal antipolar order.
             If polar_switch == 1, built a crystal polar order. */
          if (polar_switch == 1) {
            mol_coords[i_mol][0] = mol_coords[i_mol - 1][0] + a1[0] / 2.0; 
            mol_coords[i_mol][1] = mol_coords[i_mol - 1][1] + dy; 
            mol_coords[i_mol][2] = mol_coords[i_mol - 1][2] + c1[2] / 2.0; 
          }
          else {
            /* We substracte the distance between the center of mass of
               the 2 molecules (2 * temp_atm_pos), everything being computed 
                between end points. */
            mol_coords[i_mol][0] = mol_coords[i_mol - 1][0] + a1[0] / 2.0 + dx
                                   + 2.0 * temp_atm_pos[0][1];
            mol_coords[i_mol][1] = mol_coords[i_mol - 1][1] + dy; 
            mol_coords[i_mol][2] = mol_coords[i_mol - 1][2] + c1[2] / 2.0; 
          }

          /* Generate scaled position. */
          for (j = 0; j < 3; ++j){
           scaled_mol_coords[i_mol][j] = mol_coords[i_mol][j] / side[j];
           scaled_mol_coords[i_mol][j] -= NINT(scaled_mol_coords[i_mol][j]);
           mol_coords[i_mol][j] = scaled_mol_coords[i_mol][j] * side[j];
           }
          
	  /* Choose molecular orientation at random. */
          /* Choose molecular orientation at random. */
          /* If polar_switch == 2, built a crystal with antipolar order.
             If polar_switch == 1, built a crystal with polar order. */
          if (polar_switch == 1) {
            psi = 0.0;
            phi = PI / 2.0;
          }
          else {
            psi = 0.0;
            phi = 3.0 * PI / 2.0;
          }
          cos_psi = cos(psi);
          sin_psi = sin(psi);
          signe = 1.0;
          cos_theta = signe;
          sin_theta = 0.0;
          cos_phi = cos(phi);
          sin_phi = sin(phi);

          /* Get label of first atom in molecule. */
          skip = mol_first_atm[i_mol];
 
          /* Calculate atomic positions. The rotations are performed
             in the unit-cell-based reference systems, and then
             coordinates are transformed into the "standard"
             coordinate system. */
          for (i = 0; i < n_atoms_per_mol; ++i) {

            /* Calculate absolute atom label. */
            abs = skip + i;

            /* Initialize atomic position. */
            x_rot = temp_atm_pos[i][0];
            y_rot = temp_atm_pos[i][1];
            z_rot = temp_atm_pos[i][2];

            /* Rotate molecule by psi about space-fixed z axis. */
            x_tmp = cos_psi * x_rot + sin_psi * y_rot;
            y_tmp = - sin_psi * x_rot + cos_psi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Rotate molecule by theta about space-fixed x axis. */
            y_tmp = cos_theta * y_rot + sin_theta * z_rot;
            z_tmp = - sin_theta * y_rot + cos_theta * z_rot;
            y_rot = y_tmp;
            z_rot = z_tmp;

            /* Rotate molecule by phi about space-fixed z axis. */
            x_tmp = cos_phi * x_rot + sin_phi * y_rot;
            y_tmp = - sin_phi * x_rot + cos_phi * y_rot;
            x_rot = x_tmp;
            y_rot = y_tmp;

            /* Calculate relative atomic position. */
            rel_atom_coords[abs][0] = x_rot;
            rel_atom_coords[abs][1] = y_rot;
            rel_atom_coords[abs][2] = z_rot;

            /* Calculate atomic position. */
            atom_coords[abs][0] = mol_coords[i_mol][0] + x_rot;
            atom_coords[abs][1] = mol_coords[i_mol][1] + y_rot;
            atom_coords[abs][2] = mol_coords[i_mol][2] + z_rot;
          }

          /* Calculate scaled atomic coordinates for molecule i_mol. */
          scaled_atomic_coords_single_old(i_mol, n_atoms_per_mol, mol_first_atm,
             side_inv, atom_coords, scaled_atom_coords);
         
	 norm1 = norm2 = 0.0;
         for (k = 0; k < 3; ++k) {
            s[0][k] = scaled_atom_coords[skip + 1][k] -  
                      scaled_atom_coords[skip][k];  
            s[0][k] -= NINT(s[0][k]);
            u[0][k] = s[0][k] * side[k] / length;
            s[0][k] = s[0][k] * 0.5 + scaled_atom_coords[skip][k];
            s[0][k] -= NINT(s[0][k]);

            s[1][k] = scaled_atom_coords[skip + 2][k] -  
                      scaled_atom_coords[skip][k];  
            s[1][k] -= NINT(s[1][k]);
            u[1][k] = s[1][k] * side[k] / length;
            s[1][k] = s[1][k] * 0.5 + scaled_atom_coords[skip][k];
            s[1][k] -= NINT(s[1][k]);

            norm1 += u[0][k] * u[0][k];
            norm2 += u[1][k] * u[1][k];
            }

         if ((fabs(norm1 - 1.0) > 1e-05) || (fabs(norm2 - 1.0) > 1e-05)){
            printf("pb with norm.");
            exit(1);
            }

         /* Check for overlaps. */
         overlap_flag = 0;
         for (j_mol = 0; j_mol < i_mol; ++j_mol) {

           skip_j = mol_first_atm[j_mol];
           
           for (i = 0; i < 2; ++i) { 
             norm1 = 0.0;
             for (k = 0; k < 3; ++k) {
               s1[k] = scaled_atom_coords[skip_j + 1 + i][k] -  
                       scaled_atom_coords[skip_j][k];  
               s1[k] -= NINT(s1[k]);
               u1[k] = s1[k] * side[k] / length;
               s1[k] = s1[k] * 0.5 + scaled_atom_coords[skip_j][k];
               s1[k] -= NINT(s1[k]);

               norm1 += u1[k] * u1[k];
               }
             if (fabs(norm1 - 1.0) > 1e-05){
                printf("pb with norm.");
                exit(1);
             }
             for (j = 0; j < 2; ++j)
             if (overlap(s[j], u[j], s1, u1, length, 0.0, side)) {
                 printf("overlap in the close-packed crystal structure.\n");
                 printf("i_mol = %d, j_mol = %d\n", i_mol, j_mol);
                 exit(1);   
                 }
           }
         }

      /* Plot molecular configuration. */

#ifdef GRAPHICS
  if (graph_switch > 0)
          redraw(dpy, win, doubleBuffer, qobj, graph_switch, i_mol + 1,
                 mol_first_atm, n_atoms_per_mol, n_sphero_per_mol,
                 atom_coords, scaled_atom_coords, mol_coords,
                 rel_atom_coords, side, temp_bonds_1, temp_bonds_2, resph,
                 radius_sph, radius_cyl, &keep_going);
#endif

     /* Increment molecular counter. */
     ++i_mol;  
       }
     }
   } 

   /* Free memory of temporary array. */
   free_2d_array(rel_atom_coords, 3 * n_mol);

   /* Return zero if the initial condition is successfully constructed. */
   return(0);
}

/***************************************************************************/
