/* Input/output routines for builder module. */

#include "build.h"
#include "ff.h"

/*************************************************************************/

/* Read periodicity switch, number of molecular species, molecular structures,
   numbers of molecules of each species, and list of molecular species to
   header of output file. */
/*
input :
       Name of the input header file
output :
       Switch for periodic system (period_switch)
       Number of species (n_species)
       Array of number of atom in template molecule of each specie 
                                                           (n_atoms_per_mol)
       Array of template atom - atomic labels (temp_atm_lab)
       Array of template atom - atomic types (temp_atm_type)
       Array of template atom - atom branching number (temp_atm_nbr)
       Array of template atom - label of branched atoms (temp_atm_br)
       Array of template atom - order of the bonds branched to an atom
                                                            (temp_atm_ord)
       Array of template atom - realtive position of atoms in the template
                                molecule                    (temp_atm_pos)
       Array of template atom - atomic site charges (temp_atm_chg) 
       Array of number of bonds in a template molecule (n_bonds_per_mol)
       Arrays of template molecule - label of the 2 atoms sharing a bond
							(temp_bonds_1,
                                                         temp_bonds_2) 
       Array of template molecule - order of a bond (temp_bonds_ord)
       Total number of molecules (n_mols)
       Specie label of the molecules (mol_species)

*/

void read_header_direct(char *header_file, int *period_switch, int *n_species,
  int **n_atoms_per_mol, int ***temp_atm_lab, char ****temp_atm_type, 
  int ***temp_atm_nbr, int ****temp_atm_br, double ****temp_atm_ord, 
  double ****temp_atm_pos, double ***temp_atm_chg, 
  int **n_bonds_per_mol, int ***temp_bonds_1, 
  int ***temp_bonds_2, double ***temp_bonds_ord, int **n_mols_per_species, 
  int *n_mols, int *n_atoms, int **mol_species)
{
   int i, j, i_species;
   FILE *f_input;

   /* Open header file. */
   if ((f_input = fopen(header_file, "r")) == NULL)
      error_exit("Unable to open header file in read_header_direct");

   /* Read in periodicity switch. */
   fscanf(f_input,"%d\n",&(*period_switch));

   /* Read in number of molecular species. */
   fscanf(f_input,"%d\n",&(*n_species));

   /* Allocate memory for arrays containing molecular structures */
   *n_atoms_per_mol = allocate_1d_array(*n_species, sizeof(int));
   *n_mols_per_species = allocate_1d_array(*n_species, sizeof(int));
   *n_bonds_per_mol = allocate_1d_array(*n_species, sizeof(int));
   *temp_atm_lab = allocate_1d_array(*n_species, sizeof(int*));
   *temp_atm_type = allocate_1d_array(*n_species, sizeof(char**));
   *temp_atm_pos = allocate_1d_array(*n_species, sizeof(double**));
   *temp_atm_chg = allocate_1d_array(*n_species, sizeof(double*));
   *temp_atm_nbr = allocate_1d_array(*n_species, sizeof(int*));
   *temp_atm_br = allocate_1d_array(*n_species, sizeof(int**));
   *temp_atm_ord = allocate_1d_array(*n_species, sizeof(double**));
   *temp_bonds_1 = allocate_1d_array(*n_species, sizeof(int*));
   *temp_bonds_2 = allocate_1d_array(*n_species, sizeof(int*));
   *temp_bonds_ord = allocate_1d_array(*n_species, sizeof(double*));

   *n_mols = 0;
   *n_atoms = 0;
   /* Read in molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < *n_species; ++i_species) {

     /* Read in number of atoms per molecule. */
     fscanf(f_input,"%d\n", &(*n_atoms_per_mol)[i_species]);

     /* Allocate memory for template atoms arrays. */
     (*temp_atm_lab)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(int));
     (*temp_atm_type)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], F_MAX, sizeof(char));
     (*temp_atm_chg)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(double));
     (*temp_atm_nbr)[i_species] = 
        allocate_1d_array((*n_atoms_per_mol)[i_species], sizeof(int));
     (*temp_atm_pos)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], NDIM, sizeof(double));
     (*temp_atm_br)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], 6, sizeof(int));
     (*temp_atm_ord)[i_species] = 
        allocate_2d_array((*n_atoms_per_mol)[i_species], 6, sizeof(double));

     /* Read in template atom attributes. */
     for (i = 0; i < (*n_atoms_per_mol)[i_species]; ++i)
      {
      fscanf(f_input,"%d %s %d ", &(*temp_atm_lab)[i_species][i],
         (*temp_atm_type)[i_species][i], &(*temp_atm_nbr)[i_species][i]);
         for (j = 0; j < (*temp_atm_nbr)[i_species][i]; ++j)
          fscanf(f_input,"%d %lf ",&(*temp_atm_br)[i_species][i][j],
            &(*temp_atm_ord)[i_species][i][j]);
         fscanf(f_input,"\n");
         fscanf(f_input,"%lf %lf %lf %lf\n", &(*temp_atm_pos)[i_species][i][0], 
          &(*temp_atm_pos)[i_species][i][1], &(*temp_atm_pos)[i_species][i][2], 
          &(*temp_atm_chg)[i_species][i]);
/*      fscanf(f_input,"%lf %lf\n", &(*temp_atm_sigma)[i_species][i],
           &(*temp_atm_eps)[i_species][i]);   
*/
      }

     /* Read in number of bonds per molecule. */
     fscanf(f_input,"%d\n", &(*n_bonds_per_mol)[i_species]);

     if ((*n_bonds_per_mol)[i_species] > 0) {
       (*temp_bonds_1)[i_species] = 
          allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(int));
       (*temp_bonds_2)[i_species] = 
          allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(int));
       (*temp_bonds_ord)[i_species] = 
          allocate_1d_array((*n_bonds_per_mol)[i_species], sizeof(double));

       /* Read in template bond attributes. */
       for (i = 0; i < (*n_bonds_per_mol)[i_species]; ++i)
         fscanf(f_input, "%d %d %lf\n", &(*temp_bonds_1)[i_species][i], 
           &(*temp_bonds_2)[i_species][i], &(*temp_bonds_ord)[i_species][i]);
     }

     /* Read in number of molecules. */
     fscanf(f_input, "%d\n", &(*n_mols_per_species)[i_species]);

     /* Increment molecule and atom counters. */
     (*n_mols) += (*n_mols_per_species)[i_species];
     (*n_atoms) += (*n_mols_per_species)[i_species] * 
                               (*n_atoms_per_mol)[i_species]; 
   }

   /* Allocate memory for molecular species array. */
   *mol_species = allocate_1d_array(*n_mols, sizeof(int));

   /* Read in list of molecular species. */
   for (i = 0; i < *n_mols; ++i)
      fscanf(f_input, "%d ",&(*mol_species)[i]);
   fscanf(f_input, "\n");


   printf("Total number of atoms %d\n", *n_atoms);

   /* Close the input file. */
   fclose(f_input);
}

/*************************************************************************/
void write_header_direct(char *header_file, int period_switch, int n_species,
  int *n_atoms_per_mol, int **temp_atm_lab, char ***temp_atm_type,
  int **temp_atm_nbr, int ***temp_atm_br, double ***temp_atm_ord,
  double ***temp_atm_pos, double **temp_atm_chg, 
  int *n_bonds_per_mol, int **temp_bonds_1,
  int **temp_bonds_2, double **temp_bonds_ord, int *n_mols_per_species,
  int n_mols, int *mol_species)
{
   int i, j, i_species;
   FILE *f_output;

   /* Open header file. */
   if ((f_output = fopen(header_file, "w")) == NULL)
      error_exit("Unable to open header file in write_config_direct");

   /* Write out periodicity switch. */
   fprintf(f_output,"%d\n",period_switch);

   /* Write out number of molecular species. */
   fprintf(f_output,"%d\n",n_species);

   /* Write out molecular structures and number of molecules for all species.*/
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Write out number of atoms per molecule. */
      fprintf(f_output,"%d\n", n_atoms_per_mol[i_species]);

      /* Write out template atom attributes. */
      for (i = 0; i < n_atoms_per_mol[i_species]; ++i)
      {
      fprintf(f_output,"%d %s %d ", temp_atm_lab[i_species][i],
         temp_atm_type[i_species][i], temp_atm_nbr[i_species][i]);
         for (j = 0; j < temp_atm_nbr[i_species][i]; ++j)
          fprintf(f_output,"%d %g ",temp_atm_br[i_species][i][j],
            temp_atm_ord[i_species][i][j]);
         fprintf(f_output,"\n");
         fprintf(f_output,"%g %g %g %g\n", temp_atm_pos[i_species][i][0],
           temp_atm_pos[i_species][i][1], temp_atm_pos[i_species][i][2],
           temp_atm_chg[i_species][i]);
  /*    fprintf(f_output,"%g %g\n", temp_atm_sigma[i_species][i],
           temp_atm_eps[i_species][i]);
*/
      }
      /* Write out number of bonds per molecule. */
      fprintf(f_output,"%d\n", n_bonds_per_mol[i_species]);

      /* Write out template bond attributes. */
      for (i = 0; i < n_bonds_per_mol[i_species]; ++i)
      fprintf(f_output, "%d %d %g\n", temp_bonds_1[i_species][i],
           temp_bonds_2[i_species][i], temp_bonds_ord[i_species][i]);

      /* Write out number of molecules. */
      fprintf(f_output, "%d\n", n_mols_per_species[i_species]);
   }

   /* Write out list of molecular species. */
   for (i = 0; i < n_mols; ++i)
   fprintf(f_output, "%d ", mol_species[i]);
   fprintf(f_output, "\n");

   /* Close header file. */
   fclose(f_output);
}

/*************************************************************************/
/* Read atomic configuration to config file. */
/*
input :
      Name of the configuration input file (config_file)
      Switch about periodicity (period_switch)
      Total number of atoms (n_atoms)
output :
      Array of box matrix (h)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
*/

void read_config_direct(char *config_file, double ***h,
                        int n_atoms, double *length1, double *length2, 
                        double *dr_max, 
                        double *ratio_du_dr, double *dside_max,
                        double *skin, double ***atom_coords)
{
   FILE *f_input;
   int i, j;

   /* Open config file. */
   if ((f_input = fopen(config_file, "r")) == NULL)
      error_exit("Unable to open configuration file in read_config_direct");

      /* Allocate memory for the box matrix. */
      *h = allocate_2d_array(3, 3, sizeof(double));
      /* Read in box matrix elements. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fscanf(f_input, "%lf ", &(*h)[i][j]);
      fscanf(f_input, "\n");
      fscanf(f_input, "%lf %lf\n", length1, length2);
      fscanf(f_input, "%lf %lf %lf %lf\n", dr_max, ratio_du_dr, dside_max, skin);
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
      printf("   side[%d][%d] = %g\n", i, j, (*h)[i][j]);
      printf("   length1 = %g, length2 =%g\n", *length1, *length2);
      printf("   dr_max = %g\n", *dr_max);
      printf("   ratio_du_dr = %g\n", *ratio_du_dr);
      printf("   dside_max = %g\n", *dside_max);
      printf("   skin = %g\n", *skin);
      fflush(NULL);

    printf(" I am before allocating atom_coords memory n_atoms %d\n", n_atoms);

   /* Allocate memory for arrays of atomic properties. */
      *atom_coords = allocate_2d_array(n_atoms, 3, sizeof(double));

    printf(" I am after allocating atom_coords memory\n");
   /* Read in atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     fscanf(f_input, "%lf %lf %lf\n",&(*atom_coords)[i][0],
      &(*atom_coords)[i][1], &(*atom_coords)[i][2]); 

   /* Close config file. */
   fclose(f_input);
   printf("Finished reading atom_coords\n");

   fflush(NULL);

   return;
   
}

/*************************************************************************/
/* Write atomic configuration to config file. */
/*
input :
      Switch about periodicity (period_switch)
      Array of box matrix (h)
      Total number of atoms (n_atoms)
      Array of cartesian coordinates of atoms (atom_coords)
      Array of velocities of atoms (atom_vels)
output :
      Name of the configuration output file (config_file)
*/

void write_config_direct(char *config_file, double **h,
   int n_atoms, double length1, double length2, double dr_max, 
   double ratio_du_dr, double dside_max, 
   double skin, double **atom_coords)
{
   FILE *f_output;
   int i, j;
   char par_file[F_MAX];

   /* Open config file. */
/*   sprintf(par_file, "test.config"); */
   if ((f_output = fopen(config_file, "w")) == NULL)
      error_exit("Unable to open configuration file in write_config_direct");

   /* Write out box matrix if params.period_switch == 1. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fprintf(f_output, "%g ", h[i][j]);
      fprintf(f_output, "\n");
   /* Write out MC parameters. */
   fprintf(f_output, "%g %g\n", length1, length2);
   fprintf(f_output, "%g %g %g %g\n", dr_max, ratio_du_dr, dside_max, skin);

   /* Write out atomic positions and velocities. */
   for (i = 0; i < n_atoms; ++i)
     {
     fprintf(f_output, "%lf %lf %lf \n", atom_coords[i][0],
      atom_coords[i][1], atom_coords[i][2]);

     }
   /* Close config file. */
   fclose(f_output);
   fflush(NULL);
   return;
}

/****************************************************************************/
/* Read atomic positions from trajec file. */
void read_positions_direct(FILE *f_input, double **h, int n_atoms, 
    double **atom_coords)
{
   int i, j, i_atom;

   /* Read in box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fread(&h[i][j], sizeof(double), 1, f_input);

   /* Read in atomic positions. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
         fread(atom_coords[i_atom], sizeof(double), 3, f_input);
}

/****************************************************************************/
/* Write atomic positions to trajec file. */
void write_positions_direct(FILE *f_output, double **h, int n_atoms, 
    double **atom_coords)
{
   int i, j, i_atom;

   /* Write out box matrix. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j)
            fwrite(&(h[i][j]), sizeof(double), 1, f_output);

   /* Write out atomic positions. */
      for (i_atom = 0; i_atom < n_atoms; ++i_atom)
         fwrite(atom_coords[i_atom], sizeof(double), 3, f_output);
}

