/* Monte Carlo routines for spherocylinder MC program */

#include "build.h"
#include "ff.h"
#include "proto.h"

#define SMALL 1.0e-8

/* mc_cycle carries out a Monte Carlo cycle, defined as n_mol
   attempted reorientation/translation moves plus n_reshape
   constant-volume changes in unit cell shape, on average.

input: number of molecules (n_mol)
       number of unit cell reshaping moves per cycle (n_reshape)
       spherocylinder length (length)
       neighbor list skin (skin)
       array of linear dimensions of orthorhombic unit cell (h)
       maximum attempted displacement of molecule (dr_max)
       array of maximum attempted scaled displacements of molecule (ds_max)
       maximum attempted change in molecular orientation (du_max)
       unique axis for reshaping move (reshape_axis)
       maximum attempted relative change in linear dimension of unit cell 
       (dside_max)
       pointer to seed for random number generator (i_ran)
       array of scaled molecular positions (s)
       array of molecular directors (u)
       array of pointers to head of neighbor list (nl_head)
       array of pointers to tail of neighbor list (nl_tail)
       array of scaled molecular displacements since last neighbor 
        list update (ds_tot)
       array of molecular orientations at last neighbor list update (u_old)
       array of scaled molecular displacements since last neighbor list
          skin optimization (ds_opt)

output: pointer to number of attempted reorientation/translation moves 
         (n_mol_moves)
        pointer to number of accepted reorientation/translation moves 
         (n_mol_accept)
        pointer to number of attempted cell reshaping moves (n_reshape_moves)
        pointer to number of accepted cell reshaping moves (n_reshape_accept) */

void mc_cycle(int n_mols, int n_reshape, double length1, double length2, 
              double skin, 
              double **h, double **h_inv, double dr_max, double *ds_max, 
              double du_max,
              int reshape_axis, double dside_max, long *i_ran,
              nl_entry **nl_head, nl_entry **nl_tail, double **dr_tot, 
              double **dr_opt, 
              double **mol_coords, double **scaled_mol_coords, 
              double **scaled_mol_unfolded, double **atom_coords, 
              double **scaled_atom_coords, double **rel_atom_coords,
              double **mol_coords_trial, double **atom_coords_trial, 
              double **scaled_atom_coords_trial, double **s_sphero_trial,
              int *mol_first_atm, 
              int *n_atoms_per_mol, double **s_sphero, double **u_sphero, 
              int *mol_first_sphero, int *n_bonds_per_mol, int *sphero_mol,
              int cell_switch, double pressure, double *volume,
              int *n_mol_moves, int *n_mol_accept, int *n_reshape_moves, 
              int *n_reshape_accept, int *mol_species, int n_atoms, 
              int n_sphero, double *length, double **disp)
{
   int n_moves, i_move, i_type;

   /* n_moves is the total number of MC moves per cycle. */
   n_moves = n_mols + n_reshape;

   /* Loop over trial MC moves. One cycle consists of n_mol attempted
      reorientation/translation moves plus n_reshape constant-volume
      changes in unit cell shape, on average. */
   *n_mol_moves = *n_mol_accept = *n_reshape_moves = *n_reshape_accept = 0;

   for (i_move = 0; i_move < n_moves; ++i_move) {

      i_type = (int) (ran3(i_ran) * n_moves);



      if (i_type < n_mols) {
         ++(*n_mol_moves);
         (*n_mol_accept) += molecule_move(i_type, n_mols, skin, h,
            h_inv, dr_max, du_max, i_ran, nl_head, nl_tail, dr_tot, 
            dr_opt, mol_coords, scaled_mol_coords, scaled_mol_unfolded,
            atom_coords, scaled_atom_coords, atom_coords_trial, 
            scaled_atom_coords_trial, mol_first_atm, n_atoms_per_mol,  
            rel_atom_coords, s_sphero, u_sphero, mol_first_sphero, 
            n_bonds_per_mol, sphero_mol, mol_species, n_atoms, n_sphero,
            length);
      }
        else {
         ++(*n_reshape_moves);
         if (cell_switch == 0)
            (*n_reshape_accept) += reshape_move(n_mols, reshape_axis, 
               length1, length2, 
               skin, h, h_inv, dside_max, dr_max, ds_max, i_ran, 
               nl_head, nl_tail, dr_tot, dr_opt, mol_coords, scaled_mol_coords, 
               scaled_mol_unfolded, atom_coords, scaled_atom_coords,
               mol_coords_trial, atom_coords_trial, scaled_atom_coords_trial,
               s_sphero_trial, mol_first_atm, n_atoms_per_mol, rel_atom_coords, 
               s_sphero, u_sphero, mol_first_sphero, n_bonds_per_mol, 
               sphero_mol, mol_species, n_sphero, n_atoms, length);

        else
            (*n_reshape_accept) += volume_move(n_mols, reshape_axis, 
               length1, length2, 
               skin, h, h_inv, dside_max, dr_max, ds_max, pressure, 
               volume, i_ran, nl_head, nl_tail, dr_tot, dr_opt,
               mol_coords, scaled_mol_coords, scaled_mol_unfolded, atom_coords,
               scaled_atom_coords, mol_coords_trial, atom_coords_trial,
               scaled_atom_coords_trial, s_sphero_trial, mol_first_atm, 
               n_atoms_per_mol, rel_atom_coords, s_sphero, u_sphero, 
               mol_first_sphero, n_bonds_per_mol, sphero_mol, mol_species,
               n_atoms, n_sphero, length, disp);


      } 
   }

}

#undef SMALL
/****************************************************************************/

/* molecule_move attempts a simultaneous molecular translation and rotation.

input: label of molecule to move (i_mol)
       number of molecules (n_mol)
       spherocylinder length (length)
       neighbor list skin (skin)
       array of linear dimensions of orthorhombic unit cell (h)
       array of inverse linear dimensions of orthorhombic unit cell (h_inv)
       array of maximum attempted scaled displacements of molecule (ds_max)
       maximum attempted change in molecular orientation (du_max)
       pointer to seed for random number generator (i_ran)
       array of scaled molecular positions (s)
       array of molecular directors (u)
       array of pointers to head of neighbor list (nl_head)
       array of pointers to tail of neighbor list (nl_tail)
       array of scaled molecular displacements since last neighbor list update 
        (ds_tot)
       array of molecular orientations at last neighbor list update (u_old)
       array of scaled molecular displacements since last neighbor list
        skin optimization (ds_opt)

output: flag indicating whether attempted Monte Carlo move was accepted:
        1 if so, 0 if not (return value) */

int molecule_move(int i_mol, int n_mols, double skin, 
                  double **h, double **h_inv, double dr_max, 
                  double du_max, long *i_ran, 
                  nl_entry **nl_head, nl_entry **nl_tail, double **dr_tot, 
                  double **dr_opt, double **mol_coords, 
                  double **scaled_mol_coords, double **scaled_mol_unfolded, 
                  double **atom_coords, double **scaled_atom_coords,
                  double **atom_coords_trial, double **scaled_atom_coords_trial,
                  int *mol_first_atm, int *n_atoms_per_mol, 
                  double **rel_atom_coords, double **s_sphero, 
                  double **u_sphero, int *mol_first_sphero, 
                  int *n_bonds_per_mol, int *sphero_mol, int *mol_species, 
                  int n_atoms, int n_sphero, double *length)
{
   int i, j, k, j_mol, nl_flag, skip, skip_sphero, abs, abs_sphero, i_atom, 
      j_atom, t_mol;
   double half_skin, half_skin2, norm1,
      s_trial[3][3], u_trial[3][3], ds[3], dr[3], ds_side[3], 
      dr_tot_trial[4][3], 
      scaled_mol_coords_trial[3], mol_coords_trial[3], rel_trial[4][3],
      dr2_end, s1[3], u1[3];
   nl_entry *p, *p_term;
   double q[4], mag2, mag, factor, x_rot, y_rot, z_rot, 
      a11, a12, a13, a21, a22, a23, a31, a32, a33;
   /* Addition for mixture */
   int n_species, i_species;

   /* Compute various constants. */
   half_skin = 0.5 * skin;
   half_skin2 = SQR(half_skin);

   /* Get attributes of molecule i_mol. */
   i_species = mol_species[i_mol];
   skip = mol_first_atm[i_mol];
   skip_sphero = mol_first_sphero[i_mol];

   /* Get total displacement of end points of the spherocylinders 
      in molecule i_mol. */
   for (i = 0; i < n_atoms_per_mol[i_species]; ++i) { 
      abs = skip + i;
      for (k = 0; k < 3; ++k) 
         dr_tot_trial[i][k] = dr_tot[abs][k];
   }

   /* Store centers of mass of the molecules. */
   for (j = 0; j < NDIM; ++j) 
     mol_coords_trial[j] = mol_coords[i_mol][j];


   /* Generate new scaled position of i_mol. */
/*   for (j = 0; j < NDIM; ++j) {
     ds[j] = (2.0 * ran3(i_ran) - 1.0) * ds_max[j];
     ds_side[j] = ds[j] * h[j][j];
     scaled_mol_coords_trial[j] += ds[j];
     scaled_mol_coords_trial[j] -= NINT(scaled_mol_coords_trial[j]);
     }
*/
    for (j = 0; j < NDIM; ++j) {
       dr[j] = (2.0 * ran3(i_ran) - 1.0) * dr_max;
       mol_coords_trial[j] += dr[j];
     }

   /* Compute scaled displacement. */
   ds[0] = dr[0] * h_inv[0][0];
   ds[1] = dr[1] * h_inv[1][1];
   ds[2] = dr[2] * h_inv[2][2];

   /* Compute scaled position of i_mol. */
   scaled_mol_coords_trial[0] = h_inv[0][0] * mol_coords_trial[0];
   scaled_mol_coords_trial[1] = h_inv[1][1] * mol_coords_trial[1];
   scaled_mol_coords_trial[2] = h_inv[2][2] * mol_coords_trial[2];

   for (j = 0; j < NDIM; ++j) 
     scaled_mol_coords_trial[j] -= NINT(scaled_mol_coords_trial[j]);
     
   /* Recompute position of i_mol. */
   mol_coords_trial[0] = h[0][0] * scaled_mol_coords_trial[0];
   mol_coords_trial[1] = h[1][1] * scaled_mol_coords_trial[1];
   mol_coords_trial[2] = h[2][2] * scaled_mol_coords_trial[2];

   /* Generate a finite but small angular rotation.  The quaternion 
      corresponding to the unity matrix is qI = (1.0, 0.0, 0.0, 0.0). 
      We generate a quaternion close to the unity matrix. */
   do {
      mag2 = 0.0;
      for (i = 1; i < 4; ++i){
         q[i] = (2.0 * ran3(i_ran) - 1.0) * du_max;
         mag2 += SQR(q[i]);
         }
      } while (mag2 > 1.0);
   q[0] = sqrt(1 - mag2);

   /* Compute the elements of the rotation matrix. */
   a11 = SQR(q[0]) + SQR(q[1]) - SQR(q[2]) - SQR(q[3]);
   a12 = 2.0 * q[1] * q[2] + 2.0 * q[0] * q[3];
   a13 = 2.0 * q[1] * q[3] - 2.0 * q[0] * q[2];
   a21 = 2.0 * q[1] * q[2] - 2.0 * q[0] * q[3];
   a22 = SQR(q[0]) - SQR(q[1]) + SQR(q[2]) - SQR(q[3]);
   a23 = 2.0 * q[2] * q[3] + 2.0 * q[0] * q[1];
   a31 = 2.0 * q[1] * q[3] + 2.0 * q[0] * q[2];
   a32 = 2.0 * q[2] * q[3] - 2.0 * q[0] * q[1];
   a33 = SQR(q[0]) - SQR(q[1]) - SQR(q[2]) + SQR(q[3]);

   /* Calculate new relative atomic positions. The rotations are performed
      in the unit-cell-based reference systems. */

   for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

      /* Calculate absolute atom label. */
      abs = skip + i;

      /* Initialize relative atomic position. */
      x_rot = rel_atom_coords[abs][0];
      y_rot = rel_atom_coords[abs][1];
      z_rot = rel_atom_coords[abs][2];

      /* Rotate and store relative atomic positions. */
      rel_trial[i][0] = a11 * x_rot + a12 * y_rot + a13 * z_rot;
      rel_trial[i][1] = a21 * x_rot + a22 * y_rot + a23 * z_rot;
      rel_trial[i][2] = a31 * x_rot + a32 * y_rot + a33 * z_rot;

      /* Calculate atomic position. */
      atom_coords_trial[i][0] = mol_coords_trial[0] + rel_trial[i][0];
      atom_coords_trial[i][1] = mol_coords_trial[1] + rel_trial[i][1];
      atom_coords_trial[i][2] = mol_coords_trial[2] + rel_trial[i][2];

      /* Compute the displacement of spherocylinder end points and add
         to total displacement. 
         Note : i = 0 corresponds to the "bow" point of the molecule. */
      dr_tot_trial[i][0] += dr[0] + rel_trial[i][0] - x_rot;
      dr_tot_trial[i][1] += dr[1] + rel_trial[i][1] - y_rot;
      dr_tot_trial[i][2] += dr[2] + rel_trial[i][2] - z_rot;
      }

   /* Calculate scaled atomic coordinates for molecule i_mol. */
   scaled_atomic_coords_single(i_mol, n_atoms_per_mol, mol_first_atm,
                              mol_species, h_inv, atom_coords_trial, 
                               scaled_atom_coords_trial);

   periodic_boundary_conditions_single(i_mol, n_atoms_per_mol, mol_first_atm, 
                                       mol_species,h, atom_coords_trial, 
                                       scaled_atom_coords_trial);


   /* Calculate in trial s_sphero and u_sphero */
   for (i = 0; i < n_bonds_per_mol[i_species]; ++i) {

      norm1 = 0.0;
      /* Get absolute label of the spherocylinder. */
      abs_sphero = skip_sphero + i;

      /* Compute scaled center of mass and unit vector of a
         spherocylinder. */
      for (k = 0; k < NDIM; ++k) {
         s1[k] = scaled_atom_coords_trial[i + 1][k] -
                 scaled_atom_coords_trial[i][k];
         s1[k] -= NINT(s1[k]);
         u1[k] = s1[k] * h[k][k] / length[abs_sphero];
         s1[k] = s1[k] * 0.5 + scaled_atom_coords_trial[i][k];
         s1[k] -= NINT(s1[k]);

         s_trial[i][k] = s1[k];
         u_trial[i][k] = u1[k];
        norm1 += u_trial[i][k] * u_trial[i][k];
      }
           if (fabs(norm1 - 1.0) > 1e-03){
            printf("pb with norm.in molecule move\n");
            printf("norm1 = %g length %g\n", norm1, length[abs_sphero]);
            exit(1);
       }

   }

   /* Search for overlaps with neighboring molecules, and return to
      calling routine without updating molecular coordinates if an
      overlap is detected. */


     for (i = 0; i < n_bonds_per_mol[i_species]; ++i){ 
        abs_sphero = skip_sphero + i;

        if (nl_tail[abs_sphero] != NULL) {
          p = nl_head[abs_sphero];
          p_term = nl_tail[abs_sphero] -> next;
          while (p != p_term) {
            j = p -> label;
            if (overlap(s_trial[i], u_trial[i], s_sphero[j], u_sphero[j], 
                        abs_sphero, j, length, 0.0, h)){ 

                return 0;
             }
            p = p -> next; 

          }
        }
     }



     /* If the spherocylinder has moved a distance greater than half the
        thickness of the neighbor list skin since the last neighbor list
        update, check for overlaps with all other molecules and return to
        calling routine without updating molecular coordinates if an overlap
        is detected. */

   for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
      dr2_end = 0.0;
      for (k = 0; k < 3; ++k) 
         dr2_end += SQR(dr_tot_trial[i][k]);
      
       nl_flag = dr2_end > half_skin2;
       if (nl_flag)
          break;
   }

   if (nl_flag)
      for (j = 0; j < n_sphero; ++j) {
         j_mol = sphero_mol[j];
         for (i = 0; i < n_bonds_per_mol[i_species]; ++i){
              abs_sphero = skip_sphero + i;
           if (j_mol != i_mol
               && overlap(s_trial[i], u_trial[i], s_sphero[j], u_sphero[j], 
                          abs_sphero, j, length, 0.0, h)){
            return 0;
          }
        }
      }

   /* Update molecular coordinates and displacement accumulators, 
      and update neighbor lists if necessary. */
  
   /* Update atomic quantities. */ 
   for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {
       abs = skip + i;
       for (k = 0; k < NDIM; ++k) {
          atom_coords[abs][k] = atom_coords_trial[i][k];
          scaled_atom_coords[abs][k] = scaled_atom_coords_trial[i][k];
          rel_atom_coords[abs][k] = rel_trial[i][k];
          dr_opt[abs][k] += dr_tot_trial[i][k] - dr_tot[abs][k]; 
          if (!nl_flag)
             dr_tot[abs][k] = dr_tot_trial[i][k];
       }
   }

   /* Update molecular quantities. */
   for (k = 0; k < NDIM; ++k) {
      mol_coords[i_mol][k] = mol_coords_trial[k];
      scaled_mol_coords[i_mol][k] = scaled_mol_coords_trial[k];
      if (!nl_flag)
         scaled_mol_unfolded[i_mol][k] += ds[k];
   }


   /* Update spherocylinders quantities. */
   for (i = 0; i < n_bonds_per_mol[i_species]; ++i){
       abs_sphero = skip_sphero + i;
       for ( k = 0; k < NDIM; ++k){
          s_sphero[abs_sphero][k] = s_trial[i][k];
          u_sphero[abs_sphero][k] = u_trial[i][k] ;
       }
   }
   
   /* Update neighbor lists. */  
   if (nl_flag)
      update_neighbor_lists(n_mols, length, skin, h, s_sphero, u_sphero,
                            nl_head, nl_tail, n_sphero, sphero_mol, dr_tot,
                            scaled_mol_coords, scaled_mol_unfolded, n_atoms);

   /* A return value of 1 indicates that the move was accepted. */
   return 1;
}

/*****************************************************************************/
/* reshape_move attempts a change in unit cell shape with the volume 
   constrained.

input: number of molecules (n_mol)
       unique axis for reshaping move (cell_axis)
       spherocylinder length (length)
       neighbor list skin (skin)
       array of linear dimensions of orthorhombic unit cell (side)
       maximum attempted relative change in linear dimension of unit 
       cell (dside_max)
       maximum attempted displacement of molecule (dr_max)
       array of maximum attempted scaled displacements of molecule (ds_max)
       pointer to seed for random number generator (i_ran)
       array of scaled molecular positions (s)
       array of molecular directors (u)
       array of pointers to head of neighbor list (nl_head)
       array of pointers to tail of neighbor list (nl_tail)
       array of scaled molecular displacements since last neighbor 
       list update (dr_tot)
       array of molecular orientations at last neighbor list update (u_old)

output: flag indicating whether attempted Monte Carlo move was accepted:
           1 if so, 0 if not (return value) */

int reshape_move(int n_mols, int cell_axis, double length1, double length2, 
                 double skin, 
                 double **h, double **h_inv, double dside_max, 
                 double dr_max, double *ds_max, long *i_ran, 
                 nl_entry **nl_head, nl_entry **nl_tail,
                 double **dr_tot, double **dr_opt, double **mol_coords, 
                 double **scaled_mol_coords, double **scaled_mol_unfolded, 
                 double **atom_coords, double **scaled_atom_coords, 
                 double **mol_coords_trial, double **atom_coords_trial, 
                 double **scaled_atom_coords_trial, double **s_sphero_trial, 
                 int *mol_first_atm, int *n_atoms_per_mol, 
                 double **rel_atom_coords, double **s_sphero, 
                 double **u_sphero, int *mol_first_sphero, 
                 int *n_bonds_per_mol, int *sphero_mol, int *mol_species,
                 int n_sphero, int n_atoms, double *length)
{
   int i, k, i0, i1, i2, i_mol, j_mol, nl_flag;
   double half_skin, half_skin2, dh[3],
      h_perp_ave, factor, dr2_end, s1[3], disp, side;
   int i_sphero, j_sphero, skip, skip_sphero, i_atom, abs, abs_sphero;
   int i_species, j;
   double **h_trial, **h_trial_inv;

   h_trial = allocate_2d_array(3, 3, sizeof(double));
   h_trial_inv = allocate_2d_array(3, 3, sizeof(double));

   /* Compute various constants. */
   half_skin = 0.5 * skin;
   half_skin2 = SQR(half_skin);

      /* Get unit cell dimensions. */
      for (i = 0; i < 3; ++i)
         for (j = i; j < 3; ++j){
            h_trial[i][j] = h[i][j];
         }
       

   /* Determine unique axis for reshaping move and generate random change
      in unit cell dimensions, keeping the volume constant. */
   i0 = cell_axis;
   if (cell_axis == 3) {
      i0 = (int) (3.0 * ran3(i_ran));
      if (i0 == 3) i0 = 0;
   }
   i1 = (i0 + 1) % 3;
   i2 = (i0 + 2) % 3;
   dh[0] = (2.0 * ran3(i_ran) - 1.0) * dside_max;
   h_trial[i0][i0] = h[i0][i0] + dh[0];
   h_perp_ave = 0.5 * (h[i1][i1] + h[i2][i2]);
   factor = dh[0] * h[i1][i1] * h[i2][i2] / h_trial[i0][i0];
   dh[1] = dh[2] = - h_perp_ave + sqrt(SQR(h_perp_ave) - factor);
   h_trial[i1][i1] = h[i1][i1] + dh[1];
   h_trial[i2][i2] = h[i2][i2] + dh[2];

   /* Compute inverse box cell vector and compute right increment length
      of the box to be used for end points displacement. */
   for (k = 0; k < 3; ++k) {
     dh[k] = h_trial[k][k] - h[k][k];
   }
   box_dimensions(h_trial, h_trial_inv);

   /* Update molecular and atomic coordinates. */ 
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Compute new center of mass after box reshaping. */
      mol_coords_trial[i_mol][0] = h_trial[0][0] * scaled_mol_coords[i_mol][0];
      mol_coords_trial[i_mol][1] = h_trial[1][1] * scaled_mol_coords[i_mol][1];
      mol_coords_trial[i_mol][2] = h_trial[2][2] * scaled_mol_coords[i_mol][2];

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

        /* Get absolute label of the atom. */
        abs = skip + i_atom;

        /* Compute new atomic positions. Remember that the relative atom
        coordinates are unfolded. */
        atom_coords_trial[abs][0] = mol_coords_trial[i_mol][0] 
                                    + rel_atom_coords[abs][0];
        atom_coords_trial[abs][1] = mol_coords_trial[i_mol][1] 
                                    + rel_atom_coords[abs][1];
        atom_coords_trial[abs][2] = mol_coords_trial[i_mol][2] 
                                    + rel_atom_coords[abs][2];
      }
   }

   /* Compute scaled atom positions and apply periodic boundary conditions. */
     scaled_atomic_coords(n_atoms, h_trial_inv, atom_coords_trial, 
                          scaled_atom_coords_trial);
     periodic_boundary_conditions(n_atoms, h_trial, scaled_atom_coords_trial, 
                                  atom_coords_trial);


   /* Compute new center of mass for spherocylinders. */  
   for (i_mol = 0; i_mol < n_mols; ++i_mol){ 

      /* Get label of the first spherocylinder and first atom in i_mol. */
      skip_sphero = mol_first_sphero[i_mol];
      skip = mol_first_atm[i_mol];

      /* Calculate in trial s_sphero and u_sphero */
      for (i = 0; i < n_bonds_per_mol[i_species]; ++i) {

          /* Get absolute label of the spherocylinder. */
          abs_sphero = skip_sphero + i;
          abs = skip + i;

          /* Compute scaled center of mass and unit vector of a
             spherocylinder. */
          for (k = 0; k < 3; ++k) {
             s1[k] = scaled_atom_coords_trial[i + 1 + skip][k] -
                     scaled_atom_coords_trial[abs][k];
             s1[k] -= NINT(s1[k]);
             s1[k] = s1[k] * 0.5 + scaled_atom_coords_trial[abs][k];
             s1[k] -= NINT(s1[k]);

             s_sphero_trial[abs_sphero][k] = s1[k];
          }
      }
   }

   /* Check for overlaps between spherocylinders and return to calling 
      routine without updating unit cell dimensions if an overlap is found. */
   for (i_sphero = 0; i_sphero < n_sphero - 1; ++i_sphero){
      i_mol = sphero_mol[i_sphero];
      for (j_sphero = i_sphero + 1; j_sphero < n_sphero; ++j_sphero){
         j_mol = sphero_mol[j_sphero];
         if (j_mol != i_mol 
             && overlap(s_sphero_trial[i_sphero], u_sphero[i_sphero], 
                        s_sphero_trial[j_sphero], u_sphero[j_sphero], i_sphero,  
                        j_sphero, length, 0.0, h_trial))
            return 0;
      }
   }

   /* If no overlaps were found, update unit cell dimensions, volume, etc. */
   for (i = 0; i < 3; ++i) {
      h[i][i] = h_trial[i][i];
      h_inv[i][i] = h_trial_inv[i][i];
      ds_max[i] = dr_max / h[i][i];
   }

   /* Check to see whether neighbor lists need to be updated. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {
 
     /* Get label of first atom in i_mol. */
     skip = mol_first_atm[i_mol];

     /* Loop over atoms in i_mol. */
     for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

        /* Get absolute label of the atom. */
        abs = skip + i;

        dr2_end = 0.0;
        for (k = 0; k < 3; ++k) {
           disp = dh[k] * scaled_mol_unfolded[i_mol][k];
           dr_tot[abs][k] += disp;
           dr_opt[abs][k] += disp;
           dr2_end += SQR(dr_tot[abs][k]);
        }

        nl_flag = dr2_end > half_skin2;
        if (nl_flag)
           break;
     }
     if (nl_flag)
        break;
   }

   /* Update molecular and atomic coordinates. */ 
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Compute new center of mass after box reshaping. */
      mol_coords[i_mol][0] = mol_coords_trial[i_mol][0];
      mol_coords[i_mol][1] = mol_coords_trial[i_mol][1];
      mol_coords[i_mol][2] = mol_coords_trial[i_mol][2];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

         /* Get absolute label of the atom. */
         abs = skip + i_atom;

         /* Atomic positions coordinates. */
         atom_coords[abs][0] = atom_coords_trial[abs][0];
         atom_coords[abs][1] = atom_coords_trial[abs][1];
         atom_coords[abs][2] = atom_coords_trial[abs][2];

         /* Scaled atomic positions coordinates. */
         scaled_atom_coords[abs][0] = scaled_atom_coords_trial[abs][0];
         scaled_atom_coords[abs][1] = scaled_atom_coords_trial[abs][1];
         scaled_atom_coords[abs][2] = scaled_atom_coords_trial[abs][2];
      }
   }

   /* Update the center of mass of the spherocylinders. */
   for (i = 0; i < n_sphero; ++i)
      for (k = 0; k < 3; ++k)
         s_sphero[i][k] = s_sphero_trial[i][k];

   /* Update neighbor lists if necessary. */
   if (nl_flag)
      update_neighbor_lists(n_mols, length, skin, h, s_sphero, u_sphero,
                            nl_head, nl_tail, n_sphero, sphero_mol, dr_tot,
                            scaled_mol_coords, scaled_mol_unfolded, n_atoms);

   /* Exit if linear dimensions of cell become too small. */
   for (i = 0; i < 3; ++i){
      side = MAX(length1, length2);
      if (h[i][i] < (side + 2.0)) {
         fprintf(stderr, "h[%d][%d] smaller than 2*(length1+1) in reshape_move\n", i,i);
         exit(1);
      }
   }

   /* A return value of 1 indicates that the move was accepted. */
   free_2d_array(h_trial, 3 );
   free_2d_array(h_trial_inv, 3);
   return 1;
}

/*****************************************************************************/
/* volume_move attempts a change in unit cell volume at constant pressure.

input: number of molecules (n_mol)
       unique axis for reshaping move (cell_axis)
       spherocylinder length (length)
       neighbor list skin (skin)
       array of linear dimensions of orthorhombic unit cell (side)
       maximum attempted relative change in linear dimension of unit 
       cell (dside_max)
       maximum attempted displacement of molecule (dr_max)
       array of maximum attempted scaled displacements of molecule (ds_max)
       external pressure (pressure)
       pointer to unit cell volume (volume)
       pointer to seed for random number generator (i_ran)
       array of scaled molecular positions (s)
       array of molecular directors (u)
       array of pointers to head of neighbor list (nl_head)
       array of pointers to tail of neighbor list (nl_tail)
       array of scaled molecular displacements since last neighbor 
       list update (dr_tot)
       array of molecular orientations at last neighbor list update (u_old)

output: flag indicating whether attempted Monte Carlo move was accepted:
           1 if so, 0 if not (return value)
        side, ds_max, volume, e_bias, and rms_displ are modified on return. */

int volume_move(int n_mols, int cell_axis, double length1, double length2, 
                double skin, 
                double **h, double **h_inv, double dside_max, 
                double dr_max, double *ds_max, double pressure, double *volume,
                long *i_ran, nl_entry **nl_head, nl_entry **nl_tail,
                double **dr_tot, double **dr_opt, double **mol_coords, 
                double **scaled_mol_coords, double **scaled_mol_unfolded,
                double **atom_coords, double **scaled_atom_coords,
                double **mol_coords_trial, double **atom_coords_trial, 
                double **scaled_atom_coords_trial, double **s_sphero_trial,
                int *mol_first_atm, int *n_atoms_per_mol, 
                double **rel_atom_coords, double **s_sphero,
                double **u_sphero, int *mol_first_sphero,
                int *n_bonds_per_mol, int *sphero_mol, int *mol_species,
                int n_atoms, int n_sphero, double *length, double **disp)

{
   int i, k, i0, i1, i2, choice, i_mol, j_mol, nl_flag, i_sphero, j_sphero, 
      skip, skip_sphero, i_atom, abs, abs_sphero;
   double half_skin, half_skin2, 
      dh_rand, dh[3], volume_trial, dr2_end, arg, 
      boltz, s1[3];
/*   double **h_trial, **h_trial_inv; */
   double hxx, hxy, hxz, hyy, hyz, hzz;
   int i_species, j;
   nl_entry *p, *p_term;

/*   h_trial = allocate_2d_array(3, 3, sizeof(double));
   h_trial_inv = allocate_2d_array(3, 3, sizeof(double));  */

   /* Compute various constants. */
   half_skin = 0.5 * skin;
   half_skin2 = SQR(half_skin);

   /* Get unit cell dimensions. */
/*   for (i = 0; i < 3; ++i)
       for(j = i; j < 3; ++j)
      h_trial[i][j] = h[i][j]; */

     hxx = h[0][0];
     hxy = h[0][1];
     hxz = h[0][2];
     hyy = h[1][1];
     hyz = h[1][2];
     hzz = h[2][2];
     

   /* Generate random change in unit cell dimensions.
      cell_axis == -1: x, y and z dimensions of unit cell changed by 
      same amount.
      cell_axis == 0: y and z dimensions of unit cell changed by same amount.
      cell_axis == 1: x and z dimensions of unit cell changed by same amount.
      cell_axis == 2: x and y dimensions of unit cell changed by same amount.
      cell_axis == 3: x, y and z dimensions of unit cell changed independently.*/

   dh[0] = dh[1] = dh[2] = 0.0;
   dh_rand = (2.0 * ran3(i_ran) - 1.0) * dside_max;
   if (cell_axis == -1)
      dh[0] = dh[1] = dh[2] = dh_rand;
   else if (cell_axis >= 0 && cell_axis <= 2) {
      i0 = cell_axis;
      i1 = (i0 + 1) % 3;
      i2 = (i0 + 2) % 3;
      choice = (int) (2.0 * ran3(i_ran));
      if (choice == 2) choice = 0;
      if (choice == 0)
         dh[i0] = dh_rand;
      else
         dh[i1] = dh[i2] = dh_rand;
   }
   else {
      choice = (int) (3.0 * ran3(i_ran));
      if (choice == 3) choice = 0;
      dh[choice] = dh_rand;
   }

/*   for (i = 0; i < 3; ++i) {
      h_trial[i][i] = h[i][i] + dh[i];
   }
*/

   for (i = 0; i < 3; ++i) {
      h[i][i] += dh[i];
   }

/*   box_dimensions(h_trial, h_trial_inv); */
   box_dimensions(h, h_inv); 

   /* Accept new configuration based on the usual constant-pressure
      MC criterion (note that this is probably not precisely correct,
      as n_mol should be replaced by n_mol + 2/3, or something of the sort). */

   volume_trial = h[0][0] * h[1][1] * h[2][2];
   arg = - pressure * (volume_trial - *volume)
         + n_mols * log(volume_trial / *volume);

   if (arg < 0.0) {
      boltz = exp(arg);
      if (ran3(i_ran) > boltz){

            h[0][0] = hxx; h[0][1] = hxy; h[0][2] = hxz;
            h[1][1] = hyy; h[1][2] = hyz; h[2][2] = hzz;
            box_dimensions(h, h_inv);
         return 0;
    }
   }

   /* Update molecular and atomic coordinates. */ 
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Compute new center of mass after box reshaping. */
/*      mol_coords_trial[i_mol][0] = h_trial[0][0] * scaled_mol_coords[i_mol][0];
      mol_coords_trial[i_mol][1] = h_trial[1][1] * scaled_mol_coords[i_mol][1];
      mol_coords_trial[i_mol][2] = h_trial[2][2] * scaled_mol_coords[i_mol][2]; */

      mol_coords_trial[i_mol][0] = h[0][0] * scaled_mol_coords[i_mol][0];
      mol_coords_trial[i_mol][1] = h[1][1] * scaled_mol_coords[i_mol][1];
      mol_coords_trial[i_mol][2] = h[2][2] * scaled_mol_coords[i_mol][2];
      /* Get molecular species */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

        /* Get absolute label of the atom. */
        abs = skip + i_atom;

        /* Compute new atomic positions. Remember that the relative atom
        coordinates are unfolded. */
        atom_coords_trial[abs][0] = mol_coords_trial[i_mol][0] 
                                    + rel_atom_coords[abs][0];
        atom_coords_trial[abs][1] = mol_coords_trial[i_mol][1] 
                                    + rel_atom_coords[abs][1];
        atom_coords_trial[abs][2] = mol_coords_trial[i_mol][2] 
                                    + rel_atom_coords[abs][2];
      }
   }

   /* Compute scaled atom positions and apply periodic boundary conditions. */
     scaled_atomic_coords(n_atoms, h_inv, atom_coords_trial, 
                          scaled_atom_coords_trial);
     periodic_boundary_conditions(n_atoms, h, scaled_atom_coords_trial, 
                                  atom_coords_trial);


   /* Compute new center of mass for spherocylinders. */  
   for (i_mol = 0; i_mol < n_mols; ++i_mol){ 

       i_species = mol_species[i_mol];

      /* Get label of the first spherocylinder and first atom in i_mol. */
      skip_sphero = mol_first_sphero[i_mol];
      skip = mol_first_atm[i_mol];

      /* Calculate in trial s_sphero and u_sphero */
      for (i = 0; i < n_bonds_per_mol[i_species]; ++i) {

          /* Get absolute label of the spherocylinder. */
          abs_sphero = skip_sphero + i;
          abs = skip + i;

          /* Compute scaled center of mass and unit vector of a
             spherocylinder. */
          for (k = 0; k < 3; ++k) {
             s1[k] = scaled_atom_coords_trial[i + 1 + skip][k] -
                     scaled_atom_coords_trial[abs][k];
             s1[k] -= NINT(s1[k]);
             s1[k] = s1[k] * 0.5 + scaled_atom_coords_trial[abs][k];
             s1[k] -= NINT(s1[k]);

             s_sphero_trial[abs_sphero][k] = s1[k];
          }
      }
   }

   /* Check to see whether neighbor lists need to be updated. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      i_species = mol_species[i_mol];

     /* Get label of first atom in i_mol. */
     skip = mol_first_atm[i_mol];

        for (k = 0; k < 3; ++k) 
           disp[i_mol][k] = dh[k] * scaled_mol_unfolded[i_mol][k];

     /* Loop over atoms in i_mol. */
     for (i = 0; i < n_atoms_per_mol[i_species]; ++i) {

        /* Get absolute label of the atom. */
        abs = skip + i;

        dr2_end = 0.0;
        for (k = 0; k < 3; ++k) {
           dr2_end += SQR(dr_tot[abs][k] + disp[i_mol][k]);           
        }

        nl_flag = dr2_end > half_skin2;
        if (nl_flag)
           break;
     }
     if (nl_flag)
        break;
   }

   /* Check for overlaps and return to calling routine without updating 
      unit cell dimensions if an overlap is found. */

   /* Search for overlaps using neighbor list if the lists do not
      need to be updated or with a N^2 search otherwise, and return to
      calling routine without updating molecular coordinates if an
      overlap is detected. */
   if (!nl_flag){
     for (i_sphero = 0; i_sphero < n_sphero; ++i_sphero){
        if (nl_tail[i_sphero] != NULL) {
          p = nl_head[i_sphero];
          p_term = nl_tail[i_sphero] -> next;
          while (p != p_term) {
            j_sphero = p -> label;
            if (overlap(s_sphero_trial[i_sphero], u_sphero[i_sphero],
                        s_sphero_trial[j_sphero], u_sphero[j_sphero], i_sphero, 
                        j_sphero, length, 0.0, h)){

                h[0][0] = hxx; h[0][1] = hxy; h[0][2] = hxz;
                h[1][1] = hyy; h[1][2] = hyz; h[2][2] = hzz;
                box_dimensions(h, h_inv);

                return 0;
                }
            p = p -> next;
          }
        }
     }
   } else {
    for (i_sphero = 0; i_sphero < n_sphero - 1; ++i_sphero){
      i_mol = sphero_mol[i_sphero];
      for (j_sphero = i_sphero + 1; j_sphero < n_sphero; ++j_sphero){
         j_mol = sphero_mol[j_sphero];
         if (j_mol != i_mol
             && overlap(s_sphero_trial[i_sphero], u_sphero[i_sphero],
                        s_sphero_trial[j_sphero], u_sphero[j_sphero], 
                        i_sphero, j_sphero, length,
                        0.0, h)) {
            h[0][0] = hxx; h[0][1] = hxy; h[0][2] = hxz;
            h[1][1] = hyy; h[1][2] = hyz; h[2][2] = hzz;
            box_dimensions(h, h_inv);
            return 0;
            }
      }
   }
  }


   /* If no overlaps were found, update unit cell dimensions, volume, etc. */
   /*for (i = 0; i < 3; ++i) {
     h[i][i] = h_trial[i][i];
      h_inv[i][i] = h_trial_inv[i][i]; 
      ds_max[i] = dr_max / h[i][i];
   }
*/
   *volume = volume_trial;


   /* Update molecular and atomic coordinates. */ 
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Compute new center of mass after box reshaping. */
      mol_coords[i_mol][0] = mol_coords_trial[i_mol][0];
      mol_coords[i_mol][1] = mol_coords_trial[i_mol][1];
      mol_coords[i_mol][2] = mol_coords_trial[i_mol][2];

      /* Get molecular species */
      i_species = mol_species[i_mol];

      /* Get atom label offset. */
      skip = mol_first_atm[i_mol];

      /* Shift atomic positions. */
      for (i_atom = 0; i_atom < n_atoms_per_mol[i_species]; ++i_atom) {

         /* Get absolute label of the atom. */
         abs = skip + i_atom;

         /* Atomic positions coordinates. */
         atom_coords[abs][0] = atom_coords_trial[abs][0];
         atom_coords[abs][1] = atom_coords_trial[abs][1];
         atom_coords[abs][2] = atom_coords_trial[abs][2];

         /* Scaled atomic positions coordinates. */
         scaled_atom_coords[abs][0] = scaled_atom_coords_trial[abs][0];
         scaled_atom_coords[abs][1] = scaled_atom_coords_trial[abs][1];
         scaled_atom_coords[abs][2] = scaled_atom_coords_trial[abs][2];

         /* Update atomic displacement and optimization. */
         for (k = 0; k < NDIM; ++k){
            dr_tot[abs][k] += disp[i_mol][k];
            dr_opt[abs][k] += disp[i_mol][k];
         }
      }
   }

   /* Update the center of mass of the spherocylinders. */
   for (i = 0; i < n_sphero; ++i)
      for (k = 0; k < 3; ++k)
         s_sphero[i][k] = s_sphero_trial[i][k];

   /* Update neighbor lists if necessary. */
   if (nl_flag)
      update_neighbor_lists(n_mols, length, skin, h, s_sphero, u_sphero,
                            nl_head, nl_tail, n_sphero, sphero_mol, dr_tot, 
                            scaled_mol_coords, scaled_mol_unfolded, n_atoms);

   /* Exit if linear dimensions of cell become too small. */
/*   for (i = 0; i < 3; ++i)
      if (side[i] < 2.0 * (length + 1.0)) {
         fprintf(stderr, "side[%d] smaller than 2*(length+1) in volume_move\n", i);
         exit(1);
      }
*/
   /* A return value of 1 indicates that the move was accepted. */

/*   free_2d_array(h_trial, 3 );
   free_2d_array(h_trial_inv, 3); */
   return 1;
}
