/* Neighbor list routines for spherocylinder MC program */

#include "build.h"

/* update_neighbor_lists updates all neighbor lists.

input: number of molecules (n_mols)
       spherocylinder length (length)
       neighbor list skin (skin)
       array of linear dimensions of orthorhombic unit cell (h)
       array of scaled molecular positions (s)
       array of molecular directors (u)

output: array of pointers to head of neighbor list (nl_head)
        array of pointers to tail of neighbor list (nl_tail)
        array of scaled molecular displacements since last neighbor list update (dr_tot)
        array of molecular orientations at last neighbor list update (u_old) */

void update_neighbor_lists(int n_mols, double *length, 
                           double skin, double **h,
                           double **s_sphero, double **u_sphero,
                           nl_entry **nl_head, nl_entry **nl_tail,
                           int n_sphero, int *sphero_mol,
                           double **dr_tot, double **scaled_mol_coords,
                           double **scaled_mol_unfolded, int n_atoms)
{
   int i_atom, j_atom, i_sphero, j_sphero, i, skip, abs, k, 
       i_mol, j_mol, inter_flag;
   nl_entry *p;
   
   /* Purge neighbor lists. */
   for (i_sphero = 0; i_sphero < n_sphero; ++i_sphero)
      nl_tail[i_sphero] = NULL;

   /* Loop over all pairs of spherocylinders. */
   for (i_sphero = 0; i_sphero < n_sphero - 1; ++i_sphero){

      i_mol = sphero_mol[i_sphero];
      for (j_sphero = i_sphero + 1; j_sphero < n_sphero; ++j_sphero) {
         j_mol = sphero_mol[j_sphero];
         inter_flag = i_mol != j_mol;
        
         /* Check for overlap if the 2 spherocylinders belong to different
            molecules. */
         if (inter_flag) {

         /* Add an entry to the neighbor lists for spherocylinders i_sphero and 
            j_sphero if the minimum distance between spherocylinders i_sphero 
            and j_sphero is smaller than (1.0 + skin). */

         if (overlap(s_sphero[i_sphero], u_sphero[i_sphero], s_sphero[j_sphero], 
                    u_sphero[j_sphero], i_sphero, j_sphero, length, skin, h)) {

            /* Add entry to neighbor list for spherocylinder i_sphero. */
            if (nl_tail[i_sphero] == NULL)
               p = nl_head[i_sphero];
            else {
               p = nl_tail[i_sphero] -> next;
               if (p == NULL) {
                  p = gmalloc(sizeof(nl_entry));
                  p -> next = NULL;
                  nl_tail[i_sphero] -> next = p;
               }
            }
            p -> label = j_sphero;
            nl_tail[i_sphero] = p;

            /* Add entry to neighbor list for spherocylinder j_sphero. */
            if (nl_tail[j_sphero] == NULL)
               p = nl_head[j_sphero];
            else {
               p = nl_tail[j_sphero] -> next;
               if (p == NULL) {
                  p = gmalloc(sizeof(nl_entry));
                  p -> next = NULL;
                  nl_tail[j_sphero] -> next = p;
               }
            }
            p -> label = i_sphero;
            nl_tail[j_sphero] = p;
         }
         }
      }
    }

   /* Set movement accumulators for all spherocylinders. */
   for (i = 0; i < n_atoms; ++i)
      for (k = 0; k < 3; ++k) 
         dr_tot[i][k] = 0.0;

   for (i = 0; i < n_mols; ++i)
      for (k = 0; k < 3; ++k) 
         scaled_mol_unfolded[i][k] = scaled_mol_coords[i][k];
      
}
