/* Nematic order parameter routines. */

#include "build.h"

/* Calculate nematic order parameters and related quantities. */
void nematic_order(int n_mols, int n_species, int *mol_species, 
  int *n_mols_per_species, double ***mol_inert_axis, double ***mol_order_inst, 
  double **lambda, double ***v) 
{
   int i_mol, i_species, i, j, nrot, skip, abs_1, abs_2;
   double dum1, mol_order_temp[3][3], u[3], **order_temp, *lambda_temp, 
     **v_temp;

   /* Allocate memory for scratch arrays. */
   order_temp = dmatrix(1, 3, 1, 3);
   lambda_temp = dvector(1, 3);
   v_temp = dmatrix(1, 3, 1, 3);

   /* Zero instantaneous ordering tensors. */
   for (i_species = 0; i_species < n_species; ++i_species)
     for (i = 0; i < 3; ++i)
       for (j = 0; j < 3; ++j)
         mol_order_inst[i_species][i][j] = 0.0;
     
   /* Loop over molecules. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {

      /* Get label of molecular species. */
      i_species = mol_species[i_mol];

      /* The molecular director is defined as the minimum-moment 
         eigenvector of the molecular inertia tensor 
         with its polarity defined by the reference vector joining
         atoms params.mol_orient_atom_1 and params.mol_orient_atom_2.*/

      u[0] = mol_inert_axis[i_mol][2][0];
      u[1] = mol_inert_axis[i_mol][2][1];
      u[2] = mol_inert_axis[i_mol][2][2];

      /* Add contribution to instantaneous molecular ordering tensors. */
      mol_order_temp[0][0] = 1.5 * u[0] * u[0] - 0.5;
      mol_order_temp[1][1] = 1.5 * u[1] * u[1] - 0.5;
      mol_order_temp[2][2] = 1.5 * u[2] * u[2] - 0.5;
      mol_order_temp[0][1] = 1.5 * u[0] * u[1];
      mol_order_temp[0][2] = 1.5 * u[0] * u[2];
      mol_order_temp[1][2] = 1.5 * u[1] * u[2];
      mol_order_inst[i_species][0][0] += mol_order_temp[0][0];
      mol_order_inst[i_species][1][1] += mol_order_temp[1][1];
      mol_order_inst[i_species][2][2] += mol_order_temp[2][2];
      mol_order_inst[i_species][0][1] += mol_order_temp[0][1];
      mol_order_inst[i_species][0][2] += mol_order_temp[0][2];
      mol_order_inst[i_species][1][2] += mol_order_temp[1][2];
   }

   /* Loop over species. */
   for (i_species = 0; i_species < n_species; ++i_species) {

      /* Normalize instantaneous ordering tensor. */
      mol_order_inst[i_species][1][0] = mol_order_inst[i_species][0][1];
      mol_order_inst[i_species][2][0] = mol_order_inst[i_species][0][2];
      mol_order_inst[i_species][2][1] = mol_order_inst[i_species][1][2];
      dum1 = 1.0 / n_mols_per_species[i_species];
      for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j)
          mol_order_inst[i_species][i][j] *= dum1;

      /* Find eigenvalues and eigenvectors of instantaneous
         molecular ordering tensor. */
      for (i = 0; i < 3; ++i)
        for (j = 0; j < 3; ++j) 
          order_temp[i+1][j+1] = mol_order_inst[i_species][i][j];
      jacobi(order_temp, 3, lambda_temp, v_temp, &nrot);
      eigsrt(lambda_temp, v_temp, 3);

      /* Copy scratch arrays into passed arrays. */
      for (i = 0; i < 3; ++i) {
        lambda[i_species][i] = lambda_temp[i+1];
        for (j = 0; j < 3; ++j)
          v[i_species][i][j] = v_temp[i+1][j+1];
      }
   }

   /* Free up memory. */
   free_dmatrix(v_temp, 1, 3, 1, 3);
   free_dvector(lambda_temp, 1, 3);
   free_dmatrix(order_temp, 1, 3, 1, 3);
}
