/* Optimization routines for spherocylinder simulation program */

#include "build.h"
#include "ff.h"
#include "proto.h"

#define SCALE_FACTOR 1.1

/* optimize_molecule_moves adjusts dr_max to maximize the efficiency 
   (defined as the mean-squared displacement per cpu second).

input: number of molecules (n_mol)
       array of linear dimensions of orthorhombic unit cell (side)
       pointer to maximum attempted displacement of molecule (dr_max)
       pointer to previous maximum attempted displacement of 
         molecule (dr_max_old)
       array of maximum attempted scaled displacements of molecule (ds_max)
       pointer to maximum attempted change in molecular orientation (du_max)
       ratio of maximum attempted displacement to maximum reorientation 
         (ratio_du_dr)
       pointer to elapsed cpu time at last calculation of efficiency (opt_time)
       pointer to efficiency (efficiency)
       array of atomic displacements since last optimization (dr_opt)

output: dr_max, dr_max_old, ds_max, du_max, opt_time, efficiency, and dr_opt
        are modified on return. */

void optimize_molecule_moves(int n_atoms, double **h, double *dr_max, 
                             double *dr_max_old, double *ds_max, 
                             double *du_max, double ratio_du_dr,
                             double *opt_time, double *efficiency, 
                             double **dr_opt)
{
   int i;
   double efficiency_old, deriv;

   printf("\nAdjusting maximum size of molecule moves\n");

   /* Calculate efficiency (defined as the mean-squared displacement 
      per cpu second). */
   efficiency_old = *efficiency;
   *efficiency = compute_efficiency(n_atoms, opt_time, dr_opt);

   printf("   dr_max_old = %g, efficiency_old = %g / cpu s\n", *dr_max_old, 
                                                               efficiency_old);
   printf("   dr_max = %g, efficiency = %g / cpu s\n", *dr_max, *efficiency);

   /* Estimate derivative of efficiency with respect to maximum size of
      molecule moves. */
   if (*dr_max == *dr_max_old)
      deriv = 0.0;
   else
      deriv = (*efficiency - efficiency_old) / (*dr_max - *dr_max_old);

   /* Adjust maximum size of molecule moves. */
   *dr_max_old = *dr_max;
   if (deriv < 0.0)
      *dr_max /= SCALE_FACTOR;
   else
      *dr_max *= SCALE_FACTOR;
   for (i = 0; i < 3; ++i)
      ds_max[i] = *dr_max / h[i][i];
   *du_max = ratio_du_dr * (*dr_max);

   printf("   dr_max = %g\n", *dr_max);
   printf("   du_max = %g\n", *du_max);
   fflush(NULL);

   return;
}

#undef SCALE_FACTOR

#define SCALE_FACTOR 1.1

/* optimize_ratio adjusts ratio_du_dr to maximize the efficiency 
   (defined as the mean-squared displacement per cpu second).

input: number of molecules (n_mol)
       array of linear dimensions of orthorhombic unit cell (side)
       maximum attempted displacement of molecule (dr_max)
       pointer to maximum attempted change in molecular orientation (du_max)
       pointer to ratio of maximum attempted displacement to maximum 
         reorientation (ratio_du_dr)
       pointer to previous ratio of maximum attempted displacement to 
         maximum reorientation (ratio_du_dr_old)
       pointer to elapsed cpu time at last calculation of efficiency (opt_time)
       pointer to efficiency (efficiency)
       array of atomic displacements since last optimization (dr_opt). 

output: du_max, ratio_du_dr, ratio_du_dr_old, opt_time, efficiency, and dr_opt
        are modified on return. */

void optimize_ratio(int n_atoms, double **h, double dr_max, double *du_max,
                    double *ratio_du_dr, double *ratio_du_dr_old,
                    double *opt_time, double *efficiency, double **dr_opt)
{
   double efficiency_old, deriv;

   printf("\nAdjusting ratio of maximum reorientation to maximum displacement\n");

   /* Calculate efficiency (defined as the mean-squared displacement per 
      cpu second). */
   efficiency_old = *efficiency;
   *efficiency = compute_efficiency(n_atoms, opt_time, dr_opt);

   printf("   ratio_du_dr_old = %g, efficiency_old = %g / cpu s\n", 
           *ratio_du_dr_old, efficiency_old);
   printf("   ratio_du_dr = %g, efficiency = %g / cpu s\n", *ratio_du_dr, 
                                                            *efficiency);

   /* Estimate derivative of efficiency with respect to maximum size of
      molecule moves. */
   if (*ratio_du_dr == *ratio_du_dr_old)
      deriv = 0.0;
   else
      deriv = (*efficiency - efficiency_old) 
                                        / (*ratio_du_dr - *ratio_du_dr_old);

   /* Adjust maximum size of molecule moves. */
   *ratio_du_dr_old = *ratio_du_dr;
   if (deriv < 0.0)
      *ratio_du_dr /= SCALE_FACTOR;
   else
      *ratio_du_dr *= SCALE_FACTOR;
   *du_max = *ratio_du_dr * dr_max;

   printf("   ratio_du_dr = %g\n", *ratio_du_dr);
   fflush(NULL);

   return;
}

#undef SCALE_FACTOR

#define ACCEPT_MIN 0.3
#define ACCEPT_MAX 0.4
#define SCALE_FACTOR 1.1

/* optimize_reshape_moves adjusts dside_max to yield an acceptance 
   rate between ACCEPT_MIN and ACCEPT_MAX.

input: array of linear dimensions of orthorhombic unit cell (side)
       acceptance rate for unit cell reshaping moves (reshape_accept_opt)
       pointer to maximum attempted relative change in linear dimension 
       of unit cell (dside_max)

output: dside_max is modified on return. */

void optimize_reshape_moves(double **h, double reshape_accept_opt, 
                            double *dside_max)
{
   printf("\nAdjusting maximum size of cell reshaping moves\n");

   /* The acceptance rate for unit cell reshaping moves is kept 
      between ACCEPT_MIN and ACCEPT_MAX. */
   if (reshape_accept_opt < ACCEPT_MIN)
      *dside_max /= SCALE_FACTOR;
   else if (reshape_accept_opt > ACCEPT_MAX)
      *dside_max *= SCALE_FACTOR;

   printf("   reshape_accept_opt = %g\n", reshape_accept_opt);
   printf("   dside_max = %g\n", *dside_max);
   fflush(NULL);

   return;
}

#undef ACCEPT_MIN
#undef ACCEPT_MAX
#undef SCALE_FACTOR

#define SCALE_FACTOR 1.1

/* optimize_skin adjusts skin to maximize the efficiency (defined as 
   the mean-squared displacement per cpu second).

input: number of molecules (n_mol)
       spherocylinder length (length)
       array of linear dimensions of orthorhombic unit cell (side)
       pointer to current neighbor list skin (skin)
       pointer to previous neighbor list skin (skin_old)
       pointer to elapsed cpu time at last calculation of efficiency (opt_time)
       pointer to efficiency (efficiency)
       array of atomic displacements since last optimization (dr_opt)
       array of pointers to head of neighbor list (nl_head)
       array of pointers to tail of neighbor list (nl_tail)
       array of scaled molecular displacements since last neighbor list 
          update (dr_tot)

output: skin, skin_old, opt_time, efficiency, and dr_opt are modified 
        on return, and the neighbor lists and associated data structures 
        are updated. */

void optimize_skin(int n_mols, 
                   double **h, double *skin, 
                   double *skin_old, double *opt_time, double *efficiency, 
                   double **dr_opt, double **s_sphero, double **u_sphero, 
                   nl_entry **nl_head, nl_entry **nl_tail, double **dr_tot, 
                   double **scaled_mol_coords, double **scaled_mol_unfolded, 
                   int *n_bonds_per_mol, int *sphero_mol, int n_atoms, int n_sphero,
                   double *length)
{
   double efficiency_old, deriv;

   printf("\nAdjusting neighbor list skin\n");

   /* Calculate efficiency (defined as the mean-squared displacement 
      per cpu second). */
   efficiency_old = *efficiency;
   *efficiency = compute_efficiency(n_atoms, opt_time, dr_opt);

   printf("   skin_old = %g, efficiency_old = %g / cpu s\n", *skin_old, 
                                                             efficiency_old);
   printf("   skin = %g, efficiency = %g / cpu s\n", *skin, *efficiency);

   /* Estimate derivative of efficiency with respect to neighbor list skin. */
   if (*skin == *skin_old)
      deriv = 0.0;
   else
      deriv = (*efficiency - efficiency_old) / (*skin - *skin_old);

   /* Adjust neighbor list skin. */
   *skin_old = *skin;
   if (deriv < 0.0)
      *skin /= SCALE_FACTOR;
   else
      *skin *= SCALE_FACTOR;

   printf("   skin = %g\n", *skin);
   fflush(NULL);

   /* Update neighbor lists. */
   update_neighbor_lists(n_mols, length, *skin, h, s_sphero, u_sphero,
                         nl_head, nl_tail, n_sphero, sphero_mol,
                         dr_tot, scaled_mol_coords, scaled_mol_unfolded, n_atoms);
}

#undef SCALE_FACTOR

/* compute_efficiency calculates the efficiency of a simulation, defined as
   the mean-squared displacement per cpu second (i.e. a "real-time"
   diffusion constant).

input: number of molecules (n_mol)
       array of linear dimensions of orthorhombic unit cell (side)
       pointer to elapsed cpu time at last calculation of efficiency (opt_time)
       array of atomic displacements since last calculation
          of efficiency (dr_opt)

output: efficiency (return value)
        opt_time and dr_opt are also modified on return. */

double compute_efficiency(int n_atoms, double *opt_time, double **dr_opt)
{
   int i_atom, n_atom, i;
   double opt_time_old, dr2, efficiency;

   /* Calculate efficiency (defined as the mean-squared displacement per
      cpu second). */
   opt_time_old = (*opt_time);
   (*opt_time) = cpu();
   dr2 = 0.0;
   for (i_atom = 0; i_atom < n_atoms; ++i_atom)
      for (i = 0; i < 3; ++i){ 
         dr2 += SQR(dr_opt[i_atom][i]);
      }
      
   dr2 /= n_atoms;
   efficiency = dr2 / ((*opt_time) - opt_time_old);

   /* Zero displacement accumulators. */
   for (i_atom = 0; i_atom < n_atoms; ++i_atom)
      for (i = 0; i < 3; ++i)
         dr_opt[i_atom][i] = 0.0;

   return efficiency;
}
