/* Routines for calculating rms out-of-layer displacement. */

#include "build.h"

/* compute_rms_displ calculates the root-mean-square out-of-layer displacement.

input: number of layers (n_layers)
       number of molecules (n_mol)
       scaled layer position (s0)
       array of linear dimensions of orthorhombic unit cell (side)
       array of scaled molecular positions (s)

output: rms out-of-layer displacement (return value) */

double compute_rms_displ(int n_layers, int n_mols, double s0,
                         double **h, double **s)
{
   int i_mol, layer;
   double s_spacing, s0_ref, ds, sum, s_shift[3], s_displ, r_displ, rms_displ;

   /* Compute reference layer position and offset of layers from reference position. */
   s_spacing = 1.0 / n_layers;
   s0_ref = 0.5 * s_spacing - 0.5;
   ds = s0 - s0_ref;

   /* Calculate root-mean-square out-of-layer displacement. */
   sum = 0.0;
   for (i_mol = 0; i_mol < n_mols; ++i_mol) {
      s_shift[2] = s[i_mol][2] - ds;
      s_shift[2] -= NINT(s_shift[2]);
      layer = layer_assignment(n_layers, s_shift);
      s_shift[2] -= s_spacing * layer;
      s_displ = s_shift[2] - s0;
      r_displ = s_displ * h[2][2];
      sum += SQR(r_displ);
   }
   rms_displ = sqrt(sum / n_mols);

   /* Return rms displacement. */
   return rms_displ;
}

/* layer_position calculates the scaled layer position.

input: number of layers (n_layers)
       number of molecules (n_mol)
       array of linear dimensions of orthorhombic unit cell (side)
       array of scaled molecular positions (s)
       pointer to scaled layer position (s0)

output: s0 is modified upon return. */

#define TOL 1.0e-5
#define MAXITER 100

void layer_position(int n_layers, int n_mol, double *side, double **s, double *s0)
{
   int i_mol, iter, layer, keep_going;
   double s_spacing, s0_ref, ds, s_ave, s_shift[3], change;

   /* Compute reference layer position. */
   s_spacing = 1.0 / n_layers;
   s0_ref = 0.5 * s_spacing - 0.5;

   /* Iteratively refine estimate of smectic layer position. */
   iter = 0;
   do {
 
      /* Calculate shift of smectic layers from reference position. */
      ds = *s0 - s0_ref;

      /* Loop over molecules. */
      s_ave = 0.0;
      for (i_mol = 0; i_mol < n_mol; ++i_mol) {

         /* Shift molecular position by s0 - s0_ref. */
         s_shift[2] = s[i_mol][2] - ds;

         /* Apply periodic boundary conditions. */
         s_shift[2] -= NINT(s_shift[2]);

         /* Calculate layer index. */
         layer = layer_assignment(n_layers, s_shift);

         /* Collapse all layers into one layer. */
         s_shift[2] -= s_spacing * layer;

         /* Add contribution to accumulator. */
         s_ave += s_shift[2];
      }
      s_ave /= n_mol;

      /* Check for convergence. */
      change = s_ave - *s0;
      keep_going = ABS(change) > TOL;

      /* Update position of smectic layer 0. */
      *s0 = 0.5 * (s_ave + *s0);

      /* Exit if maximum number of iterations is exceeded. */
      ++iter;
      if (iter > MAXITER) {
         fprintf(stderr, "Maximum number of iterations exceeded\n");
         exit(1);
      }

   } while (keep_going);

   return;
}

#undef TOL
#undef MAXITER
