/* Optimization routines for polybeads simulation program */

#include "build.h"
#include "ff.h"
#include "proto.h"

#define SCALE_FACTOR 1.1

/* Optimize maximum size of molecule moves. The maximum size of combined
   translation/rotation moves is adjusted to maximize the efficiency
   (defined as the mean-squared displacement per cpu second).

input: 
  number of molecules (n_mol)
  array of linear dimensions of orthorhombic unit cell (side)
  pointer to maximum attempted displacement of molecule (dr_max)
  pointer to previous maximum attempted displacement of molecule (dr_max_old)
  array of maximum attempted scaled displacements of molecule (ds_max)
  pointer to maximum attempted change in molecular orientation (du_max)
  ratio of maximum attempted displacement to maximum reorientation (ratio_du_dr)
  pointer to elapsed cpu time at last calculation of efficiency (opt_time)
  pointer to efficiency (efficiency)
  array of scaled molecular displacements since last neighbor list
  skin optimization (ds_opt)

output: 
  dr_max, ds_max, du_max, opt_time, efficiency, and ds_opt are modified 
  on return. */

void optimize_molecule_moves(int n_mols, double **h, double *dr_max, 
  double *dr_max_old, double *ds_mol_max, double *dang_mol_max, 
  double ratio_dang_dr, double *opt_time, double *efficiency, double **ds_opt)
{
   int i;
   double efficiency_old, deriv;

   /* Calculate efficiency (defined as the mean-squared displacement per 
      cpu second). */
   efficiency_old = *efficiency;
   *efficiency = compute_efficiency(n_mols, h, opt_time, ds_opt);

   printf("Adjusting maximum size of molecule moves\n");
   printf("dr_max_old = %g, efficiency_old = %g / cpu s\n", *dr_max_old, efficiency_old);
   printf("dr_max = %g, efficiency = %g / cpu s\n", *dr_max, *efficiency);

   /* Estimate derivative of efficiency with respect to neighbor list skin. */
   if (*dr_max == *dr_max_old)
      deriv = 0.0;
   else
      deriv = (*efficiency - efficiency_old) / (*dr_max - *dr_max_old);

   /* Adjust maximum size of molecule moves. */
   *dr_max_old = *dr_max;
   if (deriv < 0.0)
      *dr_max /= SCALE_FACTOR;
   else
      *dr_max *= SCALE_FACTOR;
   for (i = 0; i < NDIM; ++i)
      ds_mol_max[i] = *dr_max / h[i][i];
   *dang_mol_max = ratio_dang_dr * (*dr_max);

   printf("Finished adjusting maximum size of molecule moves\n");
   printf("dr_max = %g\n", *dr_max);
   printf("ratio_dang_dr = %g\n", ratio_dang_dr);
   printf("dang_mol_max = %g\n\n", *dang_mol_max);

   return;
}

#undef SCALE_FACTOR

/*****************************************************************************/

#define SCALE_FACTOR 1.1

/* Optimize maximum ratio of angular over translational molecule moves. 
   The maximum ratio rotation/translation is adjusted to maximize the 
   efficiency (defined as the mean-squared displacement per cpu second).

input: 
  number of molecules (n_mol)
  array of linear dimensions of orthorhombic unit cell (side)
  pointer to maximum attempted displacement of molecule (dr_max)
  pointer to previous maximum attempted displacement of molecule (dr_max_old)
  array of maximum attempted scaled displacements of molecule (ds_max)
  pointer to maximum attempted change in molecular orientation (du_max)
  ratio of maximum attempted displacement to maximum reorientation (ratio_du_dr)
  pointer to elapsed cpu time at last calculation of efficiency (opt_time)
  pointer to efficiency (efficiency)
  array of scaled molecular displacements since last neighbor list
  skin optimization (ds_opt)

output: 
  dr_max, ds_max, du_max, opt_time, efficiency, and ds_opt are modified 
  on return. */

void optimize_ratio(int n_mols, double **h, double dr_max, 
  double *ratio_dang_dr_old, double *ratio_dang_dr, double *dang_mol_max, 
  double *opt_time, double *efficiency, double **ds_opt)
{
   int i;
   double efficiency_old, deriv;

   /* Calculate efficiency (defined as the mean-squared displacement per 
      cpu second). */
   efficiency_old = *efficiency;
   *efficiency = compute_efficiency(n_mols, h, opt_time, ds_opt);

   printf("Adjusting maximum ratio for molecule moves\n");
   printf("ratio_dang_dr_old = %g, efficiency_old = %g / cpu s\n", 
     *ratio_dang_dr_old, efficiency_old);
   printf("ratio_dang_dr = %g, efficiency = %g / cpu s\n", *ratio_dang_dr, 
     *efficiency);

   /* Estimate derivative of efficiency with respect to neighbor list skin. */
   if (*ratio_dang_dr == *ratio_dang_dr_old)
      deriv = 0.0;
   else
      deriv = (*efficiency - efficiency_old) / 
                                       (*ratio_dang_dr - *ratio_dang_dr_old);

   /* Adjust maximum size of molecule moves. */
   *ratio_dang_dr_old = *ratio_dang_dr;
   if (deriv < 0.0)
      *ratio_dang_dr /= SCALE_FACTOR;
   else
      *ratio_dang_dr *= SCALE_FACTOR;
   *dang_mol_max = *ratio_dang_dr * dr_max;

   printf("Finished adjusting maximum ratio for molecule moves\n");
   printf("dr_max = %g\n", dr_max);
   printf("ratio_dang_dr = %g\n", *ratio_dang_dr);
   printf("dang_mol_max = %g\n\n", *dang_mol_max);

   return;
}

#undef SCALE_FACTOR

/*****************************************************************************/
#define ACCEPT_MIN 0.3
#define ACCEPT_MAX 0.4
#define SCALE_FACTOR 1.1

/* Optimize maximum size of unit cell reshaping moves. The maximum size of 
   cell reshaping moves is adjusted to yield an acceptance rate between 
   ACCEPT_MIN and ACCEPT_MAX.

input: 
    array of linear dimensions of orthorhombic unit cell (side)
    acceptance rate for unit cell reshaping moves (reshape_accept_rate)
    pointer to maximum attempted relative change in linear dimension of 
    unit cell (dside_max)

output: dside_max is modified on return. */

void optimize_reshape_moves(double **h, double reshape_accept_rate, 
   double *dh_max)
{
   /* The acceptance rate for unit cell reshaping moves is kept 
      between ACCEPT_MIN and ACCEPT_MAX. */
   if (reshape_accept_rate < ACCEPT_MIN)
      *dh_max /= SCALE_FACTOR;
   else if (reshape_accept_rate > ACCEPT_MAX)
      *dh_max *= SCALE_FACTOR;

   printf("Adjusting maximum size of cell reshaping moves\n");
   printf("dh_max = %g\n\n", *dh_max);

   return;
}

#undef ACCEPT_MIN
#undef ACCEPT_MAX
#undef SCALE_FACTOR

/*****************************************************************************/

/* compute_efficiency calculates the efficiency of a simulation, defined as
   the mean-squared displacement per cpu second (i.e. a "real-time"
   diffusion constant).

input: 
   number of molecules (n_mol)
   array of linear dimensions of orthorhombic unit cell (side)
   pointer to elapsed cpu time at last calculation of efficiency (opt_time)
   array of scaled molecular displacements since last calculation of 
   efficiency (ds_opt)

output: 
   efficiency (return value)
   opt_time and ds_opt are also modified on return. */

double compute_efficiency(int n_mols, double **h, double *opt_time, 
   double **ds_opt)
{
   int i_mol, i;
   double opt_time_old, dr, dr2, efficiency;

   /* Calculate efficiency (defined as the mean-squared displacement per
      cpu second). */
   opt_time_old = (*opt_time);
   (*opt_time) = cpu();
   dr2 = 0.0;
   for (i_mol = 0; i_mol < n_mols; ++i_mol)
      for (i = 0; i < NDIM; ++i) {
         dr = ds_opt[i_mol][i] * h[i][i];
         dr2 += SQR(dr);
      }
   dr2 /= n_mols;
   efficiency = dr2 / ((*opt_time) - opt_time_old);
/*   printf("eff = %g, opt_time = %g, opt_time_old = %g\n",efficiency,*opt_time,
        opt_time_old);
   fflush(NULL);
*/
   /* Zero displacement accumulators. */
   for (i_mol = 0; i_mol < n_mols; ++i_mol)
      for (i = 0; i < NDIM; ++i)
         ds_opt[i_mol][i] = 0.0;

   return efficiency;
}
