/* Routine to check for overlap of two spherocylinders. */

#include "build.h"

#define SMALL 1.0e-12

/* overlap checks for overlap between two spherocylinders. If skin > 0.0, 
"near" overlaps are also detected (this feature is used in constructing 
the neighbor lists).

input: scaled position of first molecule (s1)
       director of first molecule (u1)
       scaled position of second molecule (s2)
       director of second molecule (u2)
       spherocylinder length (length)
       neighbor list skin, if applicable (skin)
       array of linear dimensions of orthorhombic unit cell (h)

output: flag indicating whether molecules are overlapping;
           1 if they are, 0 if not (return value) */

int overlap(double *s1, double *u1, double *s2, double *u2,
            int sphero1, int sphero2, double *length, double skin, double **h)
{
   int i;
   double half_length1, half_length2, dr2, ds[3], dr[3], dr_dot_u1, dr_dot_u2, u1_dot_u2, 
      denom, lambda, mu, lambda_mag, mu_mag, r_min2, r_min[3], z_ij;

   /* Compute various constants. */
   half_length1 = 0.5 * length[sphero1];
   half_length2 = 0.5 * length[sphero2];

/* printf(" length1 %g length2 %g\n", length[sphero1], length[sphero2]); */
   /* Compute pair separation. */
   dr2 = 0.0;
   for (i = 0; i < 3; ++i) {
      ds[i] = s1[i] - s2[i];
      ds[i] -= NINT(ds[i]);
      dr[i] = h[i][i] * ds[i];
      dr2 += SQR(dr[i]);
   }

   /* If the pair separation exceeds the spherocylinder length plus diameter plus
      the skin thickness, the spherocylinders don't overlap. */
   if (dr2 > SQR(half_length1 + half_length2 + 1.0 + skin)){
      return 0;
   }

   /* Test for overlaps (see Allen et al., Adv. Chem. Phys. 86, 1 (1993)). */
   dr_dot_u1 = dr[0] * u1[0] + dr[1] * u1[1] + dr[2] * u1[2];
   dr_dot_u2 = dr[0] * u2[0] + dr[1] * u2[1] + dr[2] * u2[2];
   u1_dot_u2 = u1[0] * u2[0] + u1[1] * u2[1] + u1[2] * u2[2];
   denom = 1.0 - SQR(u1_dot_u2);
   if (denom < SMALL) {
      lambda = -dr_dot_u1 / 2.0;
      mu =  dr_dot_u2 / 2.0;
   }
   else {
      lambda = (-dr_dot_u1 + u1_dot_u2 * dr_dot_u2) / denom;
      mu = ( dr_dot_u2 - u1_dot_u2 * dr_dot_u1) / denom;
   }
   lambda_mag = ABS(lambda);
   mu_mag = ABS(mu);

   if (lambda_mag > half_length1 || mu_mag > half_length2) {
printf(" lambda_mag %g mu_mag %g\n", lambda_mag, mu_mag);

/* Set lambda = L_1/2 and find the value of mu and chek for overlaps*/

         lambda = SIGN(half_length1, lambda);
         mu =  dr_dot_u2 + lambda * u1_dot_u2;
         mu_mag = ABS(mu);
         if (mu_mag > half_length2)      
         mu = SIGN(half_length2, mu);

   /* Calculate minimum separation of two spherocylinders. */
   r_min2 = 0.0;
   for (i = 0; i < 3; ++i) {
      r_min[i] = dr[i] + lambda * u1[i] - mu * u2[i];
      r_min2 += SQR(r_min[i]);
   }

   /* Return 1 if spherocylinders overlap, 0 if not. */
   return (r_min2 < SQR(1.0 + skin));
}

/* set mu = L_2/2 and find the value of lambda and chek for overlaps */

         mu = SIGN(half_length2, mu);
         lambda = -dr_dot_u1 + mu * u1_dot_u2;
         lambda_mag = ABS(lambda);
        if ( lambda_mag > half_length1) 
         lambda = SIGN(half_length1, lambda);

   /* Calculate minimum separation of two spherocylinders. */
   r_min2 = 0.0;
   for (i = 0; i < 3; ++i) {
      r_min[i] = dr[i] + lambda * u1[i] - mu * u2[i];
      r_min2 += SQR(r_min[i]);
   }

   /* Return 1 if spherocylinders overlap, 0 if not. */
   return (r_min2 < SQR(1.0 + skin));
}
    if (lambda_mag > half_length1 && mu_mag > half_length2){
            mu = SIGN(half_length2, mu);
            lambda = SIGN(half_length1, lambda);
   /* Calculate minimum separation of two spherocylinders. */
   r_min2 = 0.0;
   for (i = 0; i < 3; ++i) {
      r_min[i] = dr[i] + lambda * u1[i] - mu * u2[i];
      r_min2 += SQR(r_min[i]);
   }

   /* Return 1 if spherocylinders overlap, 0 if not. */
   return (r_min2 < SQR(1.0 + skin));
}

 }
}

#undef SMALL
